#---------------------------------------------------------------------
# Function: InstallCloud
#    Install and configure Nextcloud or Seafile based on configuration
#---------------------------------------------------------------------
InstallCloud() {
    local START_TIME=$SECONDS
    STORAGE_ROOT="/var/www"

    case $CFG_CLOUDTYPE in
	"nextcloud")
	    local INSTALL_DIR=/usr/local/lib/nextcloud
	    local CLOUD_DIR=$INSTALL_DIR/cloud	
	    InstallNextcloud
	    ;;
	"seafile")
	    echo -e "Not implemented!!!"
	    ;;
    esac
    
    # Generate SSL certificate
    systemctl stop nginx
    if [ "$CFG_SSL_NEW" == "yes" ]; then
	sh  /root/.acme.sh/acme.sh --issue --standalone -d cloud.$CFG_HOSTNAME_DOMAIN --force  >> $PROGRAMS_INSTALL_LOG_FILES 2>&1
    else
	#cp -av /home/administrator/wtlinux_server_setup/acme_ssl/arbajnok/cloud.$CFG_HOSTNAME_DOMAIN ~/.acme.sh/
	cp -av /home/$SUDO_USER/wtlinux_server_setup/acme_ssl/$CFG_HOSTNAME_DOMAIN/cloud.$CFG_HOSTNAME_DOMAIN /root/.acme.sh/ >> $PROGRAMS_INSTALL_LOG_FILES 2>&1
    fi

    systemctl restart nginx

    # Save configuration details
    cat >> /home/$SUDO_USER/wtlinux_server_setup/wtserver_configuration.txt <<EOF
# $CFG_CLOUDTYPE store the passwords
#------------------------------
URL     			: $url
$CFG_CLOUDTYPE Admin   		: $CFG_CLOUDADMINUSER
$CFG_CLOUDTYPE Password   	: $CFG_CLOUDADMINPWD

Database type			: $CFG_CLOUDDBTYPE
Database host			: $CFG_CLOUDDBHOST
Database port			: $CFG_CLOUDDBPORT
Database name			: $CFG_CLOUDDBNAME
Database user			: $CFG_CLOUDDBUSER
Database password		: $CFG_CLOUDDBPWD
#------------------------------
EOF

    MeasureTimeDuration $START_TIME
}

#---------------------------------------------------------------------
# Function: InstallNextcloud
#    Install and configure Nextcloud
#---------------------------------------------------------------------
InstallNextcloud() {
    local nextcloud_ver=31.0.0
    local nextcloud_hash=545584d128e3242337548ac2cdf31e6019b738f4
    
    echo -e "$IDENTATION_LVL_0 ${BWhite}Installing Nextcloud${NC}"

    echo -n -e "$IDENTATION_LVL_1 Installing required packages... "
    # Install required PHP packages
    apt_install php8.2 php8.2-fpm php8.2-cli php8.2-sqlite php8.2-gd php8.2-imap php8.2-curl php-pear curl \
		php8.2-dev memcached php-memcached php8.2-xml php8.2-mbstring php8.2-zip php8.2-apcu \
		php8.2-fpm php8.2-mysql php8.2-xsl php8.2-gmp php8.2-bcmath php8.2-cgi php-php-gettext libxml2-dev \
		php8.2-intl php8.2-imagick php8.2-common php8.2-opcache php8.2-readline php8.2
    echo -e "[${green}DONE${NC}]"
    
    # Enable APC before Nextcloud tools are run
    sed -i '1i apc.enabled=1\napc.enable_cli=1' /etc/php/8.2/mods-available/apcu.ini

    # Get current Nextcloud version if config.php exists
    if [ -f "$CLOUD_DIR/config.php" ]; then
	CURRENT_NEXTCLOUD_VER=$(php8.2 -r "include(\"$CLOUD_DIR/config.php\"); echo(\$CONFIG['version']);")
    else
	CURRENT_NEXTCLOUD_VER=""
    fi

    # If there is no directory or no previous version, install the base version immediately
    if [ ! -d $CLOUD_DIR ]; then
        # If there is no directory at all, install the base version
        echo -n -e "$IDENTATION_LVL_1 No existing installation found, installing Nextcloud $nextcloud_ver... "
        InstallNextcloudVer $nextcloud_ver $nextcloud_hash
    # If there is a directory but not the expected version, we will perform a version update
    elif [[ ! ${CURRENT_NEXTCLOUD_VER} =~ ^$nextcloud_ver ]]; then
	# Stop php-fpm if running. If theyre not running (which happens on a previously failed install), dont bail
	systemctl restart php8.2-fpm stop &> /dev/null || /bin/true
	# Backup existing Nextcloud installation
	# Create a backup directory to store the current installation and database to
	BACKUP_DIRECTORY=$CLOUD_DIR-backup/`date +"%Y-%m-%d-%T"`
	mkdir -p "$BACKUP_DIRECTORY"
	if [ -d $CLOUD_DIR/ ]; then
	    echo "Upgrading Nextcloud --- backing up existing installation, configuration, and database to $BACKUP_DIRECTORY..."
	    cp -r $CLOUD_DIR "$BACKUP_DIRECTORY/nextcloud-install"
 	fi
	if [ -e "$STORAGE_ROOT/nextcloud/nextcloud.db" ]; then
		cp "$STORAGE_ROOT/nextcloud/owncloud.db" "$BACKUP_DIRECTORY"
	fi
	if [ -e "$STORAGE_ROOT/nextcloud/config.php" ]; then
		cp "$STORAGE_ROOT/nextcloud/config.php" "$BACKUP_DIRECTORY"
	fi
	# Perform version-specific upgrades
	if [ -n "${CURRENT_NEXTCLOUD_VER}" ]; then
	    if [ -e "$CLOUD_DIR/config.php" ]; then
		# Remove the read-onlyness of the config, which is needed for migrations, especially for v24
		sed -i -e '/config_is_read_only/d' "$CLOUD_DIR/config.php"
	    fi
	    if [[ ${CURRENT_NEXTCLOUD_VER} =~ ^27 ]]; then
		InstallNextcloudVer 28.0.14 8a9edcfd26d318eb7d1cfa44d69796f2d1098a80
		CURRENT_NEXTCLOUD_VER="28.0.14"
	    fi
	    if [[ ${CURRENT_NEXTCLOUD_VER} =~ ^28 ]]; then
		InstallNextcloudVer 29.0.12 6cd53db7a8d730ebd5972081344db41eb2b174a8
		CURRENT_NEXTCLOUD_VER="29.0.12"
	    fi
	    if [[ ${CURRENT_NEXTCLOUD_VER} =~ ^29 ]]; then
		InstallNextcloudVer 30.0.6 c7171b03f9904a2051e765fe09ab3e95e2e564e6
		CURRENT_NEXTCLOUD_VER="30.0.6"
	    fi
	    if [[ ${CURRENT_NEXTCLOUD_VER} =~ ^30 ]]; then
		InstallNextcloudVer 31.0.0 545584d128e3242337548ac2cdf31e6019b738f4
		CURRENT_NEXTCLOUD_VER="31.0.0"
	    fi
	fi
    else
	# Ha a könyvtár létezik és a verzió is egyezik, nincs tennivaló
	echo -e "$IDENTATION_LVL_1 Nextcloud $nextcloud_ver is already installed. [${green}OK${NC}]"
    fi

    # Configure Nextcloud permissions
    cat >> ./nextcloud_permissions.sh <<EOF
#!/bin/bash

ocpath='$CLOUD_DIR'
htuser='www-data'
htgroup='www-data'
rootuser='root'

printf "Creating possible missing Directories\n"
mkdir -p \$ocpath/data
mkdir -p \$ocpath/assets
mkdir -p \$ocpath/updater

printf "chmod Files and Directories\n"
find \${ocpath}/ -type f -print0 | xargs -0 chmod 0640
find \${ocpath}/ -type d -print0 | xargs -0 chmod 0770

printf "Setting directory ownership\n"
chown -R \${rootuser}:\${htgroup} \${ocpath}/
chown -R \${htuser}:\${htgroup} \${ocpath}/apps/
chown -R \${htuser}:\${htgroup} \${ocpath}/assets/
chown -R \${htuser}:\${htgroup} \${ocpath}/config/
chown -R \${htuser}:\${htgroup} \${ocpath}/data/
chown -R \${htuser}:\${htgroup} \${ocpath}/themes/
chown -R \${htuser}:\${htgroup} \${ocpath}/updater/

chmod +x \${ocpath}/occ

printf "Setting .htaccess permissions\n"
if [ -f \${ocpath}/.htaccess ]; then
    chmod 0644 \${ocpath}/.htaccess
    chown \${rootuser}:\${htgroup} \${ocpath}/.htaccess
fi

if [ -f \${ocpath}/data/.htaccess ]; then
    chmod 0644 \${ocpath}/data/.htaccess
    chown \${rootuser}:\${htgroup} \${ocpath}/data/.htaccess
fi
EOF

    chmod +x ./nextcloud_permissions.sh
    #./nextcloud_permissions.sh

    # Create initial configuration
    instanceid=oc$(echo $CFG_HOSTNAME_DOMAIN | sha1sum | fold -w 10 | head -n 1)

    echo -n "Creating Nextcloud settings and database..."
    CFG_CLOUDDBNAME=nextcloud
    CFG_CLOUDDBUSER=nextcloud
    CFG_CLOUDDBPWD=$(dd if=/dev/urandom bs=1 count=64 2>/dev/null | sha256sum | fold -w 40 | head -n 1)
    dbruser=root
    CFG_CLOUDADMINUSER=root
    CFG_CLOUDADMINPWD=$(dd if=/dev/urandom bs=1 count=64 2>/dev/null | sha256sum | fold -w 40 | head -n 1)
    
    case $CFG_CLOUDDBTYPE in
	"mysql")
	    if [ $CFG_CLOUDDBHOST == "localhost" ]; then
		mariadb -u $dbruser -p$CFG_MYSQL_ROOT_PWD -e "CREATE DATABASE $CFG_CLOUDDBNAME" > /dev/null 2>&1
		#mysql -u $dbruser -p$CFG_MYSQL_ROOT_PWD -e "CREATE DATABASE $CFG_CLOUDDBNAME CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci" > /dev/null 2>&1
		mariadb -u $dbruser -p$CFG_MYSQL_ROOT_PWD -e "USE $CFG_CLOUDDBNAME" > /dev/null 2>&1
		mariadb -u $dbruser -p$CFG_MYSQL_ROOT_PWD -e "GRANT ALL PRIVILEGES ON $CFG_CLOUDDBNAME.* TO '$CFG_CLOUDDBUSER'@'localhost' IDENTIFIED BY '$CFG_CLOUDDBPWD'" > /dev/null 2>&1
		mysql -u $dbruser -p$CFG_MYSQL_ROOT_PWD -e "flush privileges;" > /dev/null 2>&1
	    else
		mysql -u $dbruser -p$CFG_MYSQL_ROOT_PWD -h $CFG_CLOUDDBHOST -e "CREATE DATABASE $CFG_CLOUDDBNAME" > /dev/null 2>&1
		mysql -u $dbruser -p$CFG_MYSQL_ROOT_PWD -h $CFG_CLOUDDBHOST -e "USE $CFG_CLOUDDBNAME" > /dev/null 2>&1
		mysql -u $dbruser -p$CFG_MYSQL_ROOT_PWD -h $CFG_CLOUDDBHOST -e "GRANT ALL PRIVILEGES ON $CFG_CLOUDDBNAME.* TO '$CFG_CLOUDDBUSER'@'%' IDENTIFIED BY '$CFG_CLOUDDBPWD'" > /dev/null 2>&1
		mysql -u $dbruser -p$CFG_MYSQL_ROOT_PWD -e "flush privileges;" > /dev/null 2>&1
	    fi

	    # Enable UTF8mb4 (4-byte support)
	    #databases=$(mysql -u root -p"$CFG_MYSQL_ROOT_PWD" -e "SHOW DATABASES;" | tr -d "| " | grep -v Database)
	    #for db in $databases; do
	    #    if [[ "$db" != "performance_schema" ]] && [[ "$db" != _* ]] && [[ "$db" != "information_schema" ]];
	    #    then
	    #	echo "Changing to UTF8mb4 on: $db"
	    #	mysql -u root -p"$CFG_MYSQL_ROOT_PWD" -e "ALTER DATABASE $db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"
	    #    fi
	    #done

	    # Repair and set Nextcloud config values
	    #mysqlcheck -u root -p"$CFG_MYSQL_ROOT_PWD" --auto-repair --optimize --all-databases
	    #sudo -u www-data "$STORAGE_ROOT"/owncloud/occ config:system:set mysql.utf8mb4 --type boolean --value="true"
	    #sudo -u www-data "$STORAGE_ROOT"/owncloud/occ maintenance:repair

	    sudo -u www-data php8.2 "$CLOUD_DIR"/occ maintenance:install \
		--database "mysql" \
		--database-name "$CFG_CLOUDDBNAME" \
		--database-user "$CFG_CLOUDDBUSER" \
		--database-pass "$CFG_CLOUDDBPWD" \		
		--database-host="$CFG_CLOUDDBHOST" \
		--database-port="3306" \
		--admin-user "$CFG_CLOUDADMINUSER" \
		--admin-pass "$CFG_CLOUDADMINPWD" \
		--data-dir "$STORAGE_ROOT/nextcloud" \
		--admin-email "cloud@$CFG_HOSTNAME_DOMAIN" \
		--instanceid "$instanceid"

	    # Set SMTP mail
	    #sudo -u www-data php "$CLOUD_DIR"/occ config:system:set mail_smtpmode --value="smtp"
	    # Set logrotate
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set log_rotate_size --value="10485760"
	    # Added trusted domains
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set trusted_domains 1 --value=$IP_ADDRESS
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set trusted_domains 2 --value=cloud.$CFG_HOSTNAME_DOMAIN

	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set memcache.local --value '\OC\Memcache\APCu'
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set mail_from_address --value 'cloud_administrator'
	    TIMEZONE=$(cat /etc/timezone)
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set logtimezone --value=$TIMEZONE
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set logdateformat --value 'Y-m-d H:i:s'
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set mail_domain --value=$CFG_HOSTNAME_DOMAIN
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set overwrite.cli.url --value 'https://$CFG_HOSTNAME_DOMAIN/cloud'
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set log_type --value 'syslog'
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set syslog_tag --value 'Nextcloud'
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set system_addressbook_exposed --value 'no'
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set mail_smtpmode --value 'sendmail'
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set mail_smtpauth --type boolean --value true
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set mail_smtphost --value '127.0.0.1'
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set mail_smtpport --value '587'
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set mail_smtpsecure --value ''
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set mail_smtpname --value ''
	    sudo -u www-data php "$CLOUD_DIR"/occ config:system:set mail_smtppassword --value ''

	    CONFIG_FILE="$CLOUD_DIR/config/config.php"

	    if ! grep -q "'config_is_read_only'" "$CONFIG_FILE"; then
		sed -i "/);/i \$CONFIG['config_is_read_only'] = false;" "$CONFIG_FILE"
	    fi

	    if ! grep -q "'user_backends'" "$CONFIG_FILE"; then
		sed -i "/);/i \$CONFIG['user_backends'] = array(\n  array(\n    'class' => '\\\\OCA\\\\UserExternal\\\\IMAP',\n    'arguments' => array(\n      '127.0.0.1', 143, null, null, false, false\n    ),\n  ),\n);" "$CONFIG_FILE"
	    fi

	    chown www-data.www-data $CLOUD_DIR/config.php

	    # Enable/disable apps. Note that this must be done after the Nextcloud setup.
	    # The firstrunwizard gave Josh all sorts of problems, so disabling that.
	    # user_external is what allows Nextcloud to use IMAP for login. The contacts
	    # and calendar apps are the extensions we really care about here.
	    hide_output sudo -u www-data php8.2 $CLOUD_DIR/console.php app:disable firstrunwizard
	    hide_output sudo -u www-data php8.2 $CLOUD_DIR/console.php app:enable user_external
	    hide_output sudo -u www-data php8.2 $CLOUD_DIR/console.php app:enable contacts
	    hide_output sudo -u www-data php8.2 $CLOUD_DIR/console.php app:enable calendar

	    # When upgrading, run the upgrade script again now that apps are enabled. It seems like
	    # the first upgrade at the top won't work because apps may be disabled during upgrade?
	    # Check for success (0=ok, 3=no upgrade needed).
	    sudo -u www-data php8.2 $CLOUD_DIR/occ upgrade
	    E=$?
	    if [ $E -ne 0 ] && [ $E -ne 3 ]; then exit 1; fi

	    # Disable default apps that are not needed
            sudo -u www-data php8.2 $CLOUD_DIR/occ app:disable photos dashboard activity weather_status logreader | (grep -v "No such app enabled" || /bin/true)

	    # Install additional apps
	    (sudo -u www-data php8.2 $CLOUD_DIR/occ app:install notes) || true
	    hide_output sudo -u www-data php8.2 $CLOUD_DIR/console.php app:enable notes

	    (sudo -u www-data php8.2 $CLOUD_DIR/occ app:install twofactor_totp) || true
	    hide_output sudo -u www-data php8.2 $CLOUD_DIR/console.php app:enable twofactor_totp

	    # Upgrade all apps
	    sudo -u www-data php $CLOUD_DIR/occ app:update --all

	    echo
	    echo "Nextcloud version:"
	    sudo -u www-data php8.2 "$CLOUD_DIR"/occ status
	    sleep 3
	    echo
	;;
	"sqlite")
	    # Setup Nextcloud if the Nextcloud database does not yet exist. Running setup when
	    # the database does exist wipes the database and user data.
	    #if [ ! -f $STORAGE_ROOT/owncloud/owncloud.db ]; then
		# Create user data directory
		#mkdir -p $STORAGE_ROOT/owncloud
	    
		# Set permissions
		#chown -R www-data.www-data $STORAGE_ROOT/owncloud $STORAGE_ROOT/owncloud

		# Execute Nextcloud's setup step, which creates the Nextcloud sqlite database.
		# It also wipes it if it exists. And it updates config.php with database
		# settings and deletes the autoconfig.php file.
		#    (cd $STORAGE_ROOT/owncloud; sudo -u www-data php $STORAGE_ROOT/owncloud/index.php;)
	    #fi
	    
		# Migrate users_external data from <0.6.0 to version 3.0.0
		# (see https://github.com/nextcloud/user_external).
		# This version was probably in use in Mail-in-a-Box v0.41 (February 26, 2019) and earlier.
		# We moved to v0.6.3 in 193763f8. Ignore errors - maybe there are duplicated users with the
		# correct backend already.
	    sqlite3 "$STORAGE_ROOT/owncloud/owncloud.db" "UPDATE oc_users_external SET backend='127.0.0.1';" || /bin/true
	;;
    esac

    # Set PHP FPM values to support large file uploads
    # (semicolon is the comment character in this file, hashes produce deprecation warnings)
    #sed -i "s/upload_max_filesize =.*/upload_max_filesize = 16G/g" /etc/php/8.2/fpm/php.ini
    sed -i "s/upload_max_filesize =.*/upload_max_filesize = 1G/g" /etc/php/8.2/fpm/php.ini
    #sed -i "s/post_max_size =.*/post_max_size = 16G/g" /etc/php/8.2/fpm/php.ini
    sed -i "s/post_max_size =.*/post_max_size = 1G/g" /etc/php/8.2/fpm/php.ini
    sed -i "s/output_buffering =.*/output_buffering = 16384/g" /etc/php/8.2/fpm/php.ini
    sed -i "s/memory_limit =.*/memory_limit = 512M/g" /etc/php/8.2/fpm/php.ini
    sed -i "s/max_execution_time =.*/max_execution_time = 600/g" /etc/php/8.2/fpm/php.ini
    sed -i "s/short_open_tag =.*/short_open_tag = On/g" /etc/php/8.2/fpm/php.ini

    #management/editconf.py /etc/php/8.2/fpm/php.ini -c ';' \
    #	upload_max_filesize=16G \
    #	post_max_size=16G \
    #	output_buffering=16384 \
    #	memory_limit=512M \
    #	max_execution_time=600 \
    #	short_open_tag=On

    # Set Nextcloud recommended opcache settings
    #if grep -q opcache.enable=0 /etc/php/7.3/mods-available/apcu.ini; then
    #    sed -i "s/;opcache.enable=.*/opcache.enable=1/g" /etc/php/7.3/fpm/php.ini
    #fi
    sed -i "s/;opcache.enable.*/opcache.enable=1/g" /etc/php/8.2/cli/conf.d/10-opcache.ini
    sed -i "s/;opcache.enable_cli.*/opcache.enable_cli=1/g" /etc/php/8.2/cli/conf.d/10-opcache.ini
    sed -i "s/;opcache.interned_strings_buffer.*/opcache.interned_strings_buffer=16/g" /etc/php/8.2/cli/conf.d/10-opcache.ini
    sed -i "s/max_accelerated_files=.*/max_accelerated_files=10000/g" /etc/php/8.2/cli/conf.d/10-opcache.ini
    sed -i "s/opcache.memory_consumption=.*/opcache.memory_consumption=128/g" /etc/php/8.2/cli/conf.d/10-opcache.ini
    sed -i "s/;opcache.save_comments=.*/opcache.save_comments=1/g" /etc/php/8.2/cli/conf.d/10-opcache.ini
    sed -i "s/opcache.revalidate_freq=.*/opcache.revalidate_freq=1/g" /etc/php/8.2/cli/conf.d/10-opcache.ini

    #management/editconf.py /etc/php/8.2/cli/conf.d/10-opcache.ini -c ';' \
    #	opcache.enable=1 \
    #	opcache.enable_cli=1 \
    #	opcache.interned_strings_buffer=16 \
    #	opcache.max_accelerated_files=10000 \
    #	opcache.memory_consumption=128 \
    #	opcache.save_comments=1 \
    #	opcache.revalidate_freq=1

   # Configure the path environment for php-fpm
   sed -i '/env\[PATH/s/;//g' /etc/php/8.2/fpm/pool.d/www.conf

   # If apc is explicitly disabled we need to enable it
   #if grep -q apc.enabled=0 /etc/php/8.2/mods-available/apcu.ini; then
   #if grep -q apc.enabled=1 /etc/php/8.2/mods-available/apcu.ini; then
   #    echo "apc.enabled=1" >> /etc/php/8.2/mods-available/apcu.ini
   #fi

   # Set up a general cron job for Nextcloud.
   # Also add another job for Calendar updates, per advice in the Nextcloud docs
   # https://docs.nextcloud.com/server/24/admin_manual/groupware/calendar.html#background-jobs
   cat > /etc/cron.d/cron-nextcloud << EOF;
#!/bin/bash
# Wolftronics server setup script
*/5 * * * *	root	sudo -u www-data php8.2 -f $CLOUD_DIR/cron.php
*/5 * * * *	root	sudo -u www-data php8.2 -f $CLOUD_DIR/occ dav:send-event-reminders
EOF
    chmod +x /etc/cron.d/cron-nextcloud

    # We also need to change the sending mode from background-job to occ
    # Or else the reminders will just be sent as soon as possible when the background jobs run
    hide_output sudo -u www-data php8.2 -f $CLOUD_DIR/occ config:app:set dav sendEventRemindersMode --value occ

    # Run the maintenance command
    hide_output sudo -u www-data php8.2 $CLOUD_DIR/occ maintenance:repair --include-expensive

    # Set the config to read-only
    sed -i'' "s/'config_is_read_only'\s*=>\s*false/'config_is_read_only' => true/" "$STORAGE_ROOT/owncloud/config.php"

    # Create nextcloud log in /var/log
    hide_output install -m 644 conf/rsyslog/20-nextcloud.conf /etc/rsyslog.d/

    # There's nothing much of interest that a user could do as an admin for Nextcloud,
    # and there's a lot they could mess up, so we don't make any users admins of Nextcloud.
    # But if we wanted to, we would do this:
    # ```
    # for user in $(management/cli.py user admins); do
    #	 sqlite3 $STORAGE_ROOT/owncloud/owncloud.db "INSERT OR IGNORE INTO oc_group_user VALUES ('admin', '$user')"
    # done
    # ```

    # Enable PHP modules and restart PHP.
    restart_service php8.2-fpm

#echo -n "Configuring Redis Cache for nextCloud..."
#hide_output add-apt-repository -y ppa:chris-lea/redis-server
#hide_output apt-get update
#apt_get_quiet upgrade
#apt_install redis-server
#systemctl restart apache2
#systemctl restart nginx
#phpenmod redis
#hide_output systemctl start redis-server
#hide_output systemctl enable redis-server
#apt_install php-redis

#cat >> $STORAGE_ROOT/owncloud/config/config.php <<EOF
#'memcache.distributed' => '\OC\Memcache\Redis',
#'memcache.local' => '\OC\Memcache\Redis',
#'memcache.locking' => '\OC\Memcache\Redis',
#'redis' => array(
#     'host' => 'localhost',
#     'port' => 6379,
#     ),
#EOF
}

#---------------------------------------------------------------------
# Function: InstallNextcloudVer
#    Install a specific version of Nextcloud
#---------------------------------------------------------------------
InstallNextcloudVer() {
    version=$1
    hash=$2

    echo -n -e "$IDENTATION_LVL_1 Installing Nextcloud $version "

    # Remove the current Nextcloud installation
    rm -rf $CLOUD_DIR

    # Download and verify the Nextcloud package
    wget_verify https://download.nextcloud.com/server/releases/nextcloud-$version.zip $hash /tmp/nextcloud.zip

    # Extract Nextcloud
    unzip -q /tmp/nextcloud.zip -d $INSTALL_DIR
    mv $INSTALL_DIR/nextcloud $CLOUD_DIR
    rm -f /tmp/nextcloud.zip

    # Empty the skeleton dir to save some space for each new user
    rm -rf $CLOUD_DIR/core/skeleton/*
    echo -e "[${green}DONE${NC}]"

    echo -n -e "$IDENTATION_LVL_1 Installing Nextcloud Apps"
    # Create the apps directory if it doesn't exist
    mkdir -p $CLOUD_DIR/apps

    # Install additional Nextcloud apps
    InstallNextcloudApps
    echo -e "[${green}DONE${NC}]"

    # Fix directory permissions
    chmod 750 $CLOUD_DIR/{apps,config}

    # Create a symlink to the config.php in STORAGE_ROOT (for upgrades we're restoring the symlink we previously
    # put in, and in new installs we're creating a symlink and will create the actual config later).
    ln -sf "$STORAGE_ROOT/nextcloud/config.php" $CLOUD_DIR/config/config.php

    # Make sure permissions are correct or the upgrade step won't run.
    # $STORAGE_ROOT/owncloud may not yet exist, so use -f to suppress
    # that error.
    chown -f -R www-data:www-data "$STORAGE_ROOT/nextcloud" $INSTALL_DIR || /bin/true

    # If this isn't a new installation, immediately run the upgrade script.
    # Then check for success (0=ok and 3=no upgrade needed, both are success).
    if [ -e $STORAGE_ROOT/nextcloud/nextcloud.db ]; then
	sudo -u www-data php $CLOUD_DIR/occ upgrade
	if [ \( $? -ne 0 \) -a \( $? -ne 3 \) ]; then
	    echo "Trying NextCloud upgrade again to work around NextCloud upgrade bug..."
	    sudo -u www-data php $CLOUD_DIR/occ upgrade
            if [ \( $? -ne 0 \) -a \( $? -ne 3 \) ]; then 
                exit 1
            fi
	    sudo -u www-data php $CLOUD_DIR/occ maintenance:mode --off
	    echo "...which seemed to work."
	fi

	# Add missing indices. NextCloud didn't include this in the normal upgrade because it might take some time.
	sudo -u www-data php8.2 $CLOUD_DIR/occ db:add-missing-indices
	sudo -u www-data php8.2 $CLOUD_DIR/occ db:add-missing-primary-keys

	# Run conversion to BigInt identifiers, this process may take some time on large tables.
	sudo -u www-data php8.2 $CLOUD_DIR/occ db:convert-filecache-bigint --no-interaction
    fi

    # Configure web server (Apache or Nginx)
    case $CFG_WEBSERVER in
    "apache")
	cat >> /etc/apache2/sites-available/nextcloud.conf <<EOF
<VirtualHost *:80>
 DocumentRoot "$CLOUD_DIR"
 ServerName cloud.$CFG_HOSTNAME_DOMAIN

 ErrorLog ${APACHE_LOG_DIR}/error.log
 CustomLog ${APACHE_LOG_DIR}/access.log combined

<Directory $CLOUD_DIR/>
 Options +FollowSymlinks
 AllowOverride All

 <IfModule mod_dav.c>
 Dav off
 </IfModule>

 SetEnv HOME $CLOUD_DIR/>
 SetEnv HTTP_HOME $CLOUD_DIR
 Satisfy Any

</Directory>

</VirtualHost>
EOF

	cat >> /etc/apache2/sites-available/nextcloud-le-ssl.conf <<EOF
<IfModule mod_headers.c>
     Header always set Strict-Transport-Security "max-age=15768000; preload"
 </IfModule>
EOF
	
	ln -s /etc/apache2/sites-available/nextcloud.conf /etc/apache2/sites-enabled/nextcloud.conf
	a2enmod rewrite headers env dir mime setenvif ssl
	a2ensite default-ssl
	systemctl restart apache2
	;;
    "nginx")
	cat > /etc/nginx/sites-available/nextcloud.vhost <<EOF;
server {
    listen 80;
    server_name cloud.$CFG_HOSTNAME_DOMAIN;
    return 301 https://\$server_name\$request_uri;
}

server {
#    listen 80;
    listen 443 ssl http2;
    server_name cloud.$CFG_HOSTNAME_DOMAIN;

    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_certificate     /home/$SUDO_USER/.acme.sh/cloud.$CFG_HOSTNAME_DOMAIN/cloud.$CFG_HOSTNAME_DOMAIN.cer;
    ssl_certificate_key /home/$SUDO_USER/.acme.sh/cloud.$CFG_HOSTNAME_DOMAIN/cloud.$CFG_HOSTNAME_DOMAIN.key;

    ssl_session_timeout 1d;
    ssl_session_cache shared:SSL:10m;
    ssl_session_tickets off;

    # Add headers to serve security related headers
    # Before enabling Strict-Transport-Security headers please read into this
    # topic first.
    add_header Strict-Transport-Security "max-age=15768000;preload" always;
    add_header X-Content-Type-Options nosniff;
    add_header X-Frame-Options "SAMEORIGIN";
    add_header X-XSS-Protection "1; mode=block";
    add_header X-Robots-Tag none;
    add_header X-Download-Options noopen;
    add_header X-Permitted-Cross-Domain-Policies none;

    # Path to the root of your installation
    root $CLOUD_DIR/;

    location = /robots.txt {
        allow all;
        log_not_found off;
        access_log off;
    }

    # The following 2 rules are only needed for the user_webfinger app.
    # Uncomment it if you're planning to use this app.
    #rewrite ^/.well-known/host-meta /public.php?service=host-meta last;
    #rewrite ^/.well-known/host-meta.json /public.php?service=host-meta-json
    # last;

    location = /.well-known/carddav {
        return 301 \$scheme://\$host/remote.php/dav;
    }
    location = /.well-known/caldav {
       return 301 \$scheme://\$host/remote.php/dav;
    }

    location ~ /.well-known/acme-challenge {
      allow all;
    }

    # set max upload size
    client_max_body_size 512M;
    fastcgi_buffers 64 4K;

    # Disable gzip to avoid the removal of the ETag header
    gzip off;

    # Uncomment if your server is build with the ngx_pagespeed module
    # This module is currently not supported.
    #pagespeed off;

    error_page 403 /core/templates/403.php;
    error_page 404 /core/templates/404.php;

    location / {
       rewrite ^ /index.php\$uri;
    }

    location ~ ^/(?:build|tests|config|lib|3rdparty|templates|data)/ {
       deny all;
    }
    location ~ ^/(?:\.|autotest|occ|issue|indie|db_|console) {
       deny all;
     }

    location ~ ^/(?:index|remote|public|cron|core/ajax/update|status|ocs/v[12]|updater/.+|ocs-provider/.+|core/templates/40[34])\.php(?:$|/) {
       include fastcgi_params;
       fastcgi_split_path_info ^(.+\.php)(/.*)$;
       fastcgi_param SCRIPT_FILENAME \$document_root\$fastcgi_script_name;
       fastcgi_param PATH_INFO \$fastcgi_path_info;
       #Avoid sending the security headers twice
       fastcgi_param modHeadersAvailable true;
       fastcgi_param front_controller_active true;
       fastcgi_pass unix:/run/php/php7.3-fpm.sock;
       fastcgi_intercept_errors on;
       fastcgi_request_buffering off;
    }

    location ~ ^/(?:updater|ocs-provider)(?:$|/) {
       try_files \$uri/ =404;
       index index.php;
    }

    # Adding the cache control header for js and css files
    # Make sure it is BELOW the PHP block
    location ~* \.(?:css|js)$ {
        try_files \$uri /index.php\$uri\$is_args\$args;
        add_header Cache-Control "public, max-age=7200";
        # Add headers to serve security related headers (It is intended to
        # have those duplicated to the ones above)
        add_header X-Content-Type-Options nosniff;
        add_header X-Frame-Options "SAMEORIGIN";
        add_header X-XSS-Protection "1; mode=block";
        add_header X-Robots-Tag none;
        add_header X-Download-Options noopen;
        add_header X-Permitted-Cross-Domain-Policies none;
        # Optional: Don't log access to assets
        access_log off;
   }

   location ~* \.(?:svg|gif|png|html|ttf|woff|ico|jpg|jpeg)$ {
        try_files \$uri /index.php\$uri\$is_args\$args;
        # Optional: Don't log access to other assets
        access_log off;
   }
}
EOF

	mkdir -p /etc/nginx/sites-enabled
	ln -s /etc/nginx/sites-available/nextcloud.vhost /etc/nginx/sites-enabled/nextcloud.vhost
	;;
    esac
}

InstallNextcloudApps() {
    # Admin
    # https://apps.nextcloud.com/apps/onlyoffice
    #wget_verify https://github.com/ONLYOFFICE/onlyoffice-nextcloud/releases/download/v9.6.0/onlyoffice.tar.gz fce197299377535b64097e86458cd9b2a8f98cf0 /tmp/onlyoffice.tgz
    #tar xf /tmp/onlyoffice.tgz -C $CLOUD_DIR/apps/
    #rm /tmp/onlyoffice.tgz

    # https://apps.nextcloud.com/apps/richdocuments
    wget_verify https://github.com/nextcloud-releases/richdocuments/releases/download/v8.6.1/richdocuments-v8.6.1.tar.gz c5db28abb1481f74fd7e80539d322f81a3327dcb /tmp/richdocuments.tgz
    tar xf /tmp/richdocuments.tgz -C $CLOUD_DIR/apps/
    rm /tmp/richdocuments.tgz
    
    # https://apps.nextcloud.com/apps/drop_account
    wget_verify https://packages.framasoft.org/projects/nextcloud-apps/drop-account/drop_account-2.7.1.tar.gz b14b9011f85ba25ce13ff4cedd1e50e96bcffdec /tmp/drop_account.tgz
    tar xf /tmp/drop_account.tgz -C $CLOUD_DIR/apps/
    rm /tmp/drop_account.tgz

    # https://apps.nextcloud.com/apps/files_accesscontrol
    wget_verify https://github.com/nextcloud-releases/files_accesscontrol/releases/download/v2.0.0/files_accesscontrol-v2.0.0.tar.gz 5b8a2c489cb36bed2cd490c1517506612cdfb804 /tmp/files_accesscontrol.tgz
    tar xf /tmp/files_accesscontrol.tgz -C $CLOUD_DIR/apps/
    rm /tmp/files_accesscontrol.tgz
    
    # https://apps.nextcloud.com/apps/files_antivirus
    wget_verify https://github.com/nextcloud-releases/files_antivirus/releases/download/v6.0.0-beta.1/files_antivirus-v6.0.0-beta.1.tar.gz dc44dc2d9459050cd3a10464830f62b25772c9e1 /tmp/files_antivirus.tgz
    tar xf /tmp/files_antivirus.tgz -C $CLOUD_DIR/apps/
    rm /tmp/files_antivirus.tgz

    # https://apps.nextcloud.com/apps/ransomware_protection
    #wget_verify https://github.com/nextcloud/ransomware_protection/releases/download/v1.5.1/ransomware_protection-1.5.1.tar.gz 77d05b65a049dfbf29a3a4c7fa70903ef10fdb6e /tmp/ransomware_protection.tgz
    #tar xf /tmp/ransomware_protection.tgz -C $CLOUD_DIR/apps/
    #rm /tmp/ransomware_protection.tgz

    # Admin apps
    # https://apps.nextcloud.com/apps/sociallogin
    # 'social_login_auto_redirect' => true setting in config.php
    wget_verify https://github.com/zorn-v/nextcloud-social-login/releases/download/v5.9.2/release.tar.gz 6d44480bd798e40489c5fea716a1f214c803bf79 /tmp/sociallogin.tgz
    tar xf /tmp/sociallogin.tgz -C $CLOUD_DIR/apps/
    rm /tmp/sociallogin.tgz

    # https://apps.nextcloud.com/apps/external
    wget_verify https://github.com/nextcloud-releases/external/releases/download/v6.0.1/external-v6.0.1.tar.gz d8b5ec2c9911ebf4b8a5a6cb53b1f54336c0a4ec /tmp/external.tgz
    tar xf /tmp/external.tgz -C $CLOUD_DIR/apps/
    rm /tmp/external.tgz

    # https://apps.nextcloud.com/apps/sharepoint
    wget_verify https://github.com/nextcloud-releases/sharepoint/releases/download/v1.19.0/sharepoint-v1.19.0.tar.gz 108cbeb8a018631d1723a9a2e46411feeb42f60d /tmp/sharepoint.tgz
    tar xf /tmp/sharepoint.tgz -C $CLOUD_DIR/apps/
    rm /tmp/sharepoint.tgz

    # https://apps.nextcloud.com/apps/announcementcenter
    wget_verify https://github.com/nextcloud-releases/announcementcenter/releases/download/v7.1.0/announcementcenter-v7.1.0.tar.gz acd361da895a91d3cf71fe3569287dc2e90e0842 /tmp/announcementcenter.tgz
    tar xf /tmp/announcementcenter.tgz -C $CLOUD_DIR/apps/
    rm /tmp/announcementcenter.tgz

    # https://apps.nextcloud.com/apps/impersonate
    wget_verify https://github.com/nextcloud-releases/impersonate/releases/download/v2.0.0/impersonate-v2.0.0.tar.gz a67366f5de79be2dc6f71ee0ce5cf027555e90f9 /tmp/impersonate.tgz
    tar xf /tmp/impersonate.tgz -C $CLOUD_DIR/apps/
    rm /tmp/impersonate.tgz

    # https://apps.nextcloud.com/apps/ocdownloader
    #wget_verify https://github.com/e-alfred/ocdownloader/releases/download/1.6.0-Alpha2/ocdownloader_1.6.0alpha2 59625f5ec936ab1c26744f222718124848944d9d /tmp/ocdownloader.tgz
    #tar xf /tmp/ocdownloader.tgz -C $CLOUD_DIR/apps/
    #rm /tmp/ocdownloader.tgz

    # https://apps.nextcloud.com/apps/files_retention
    wget_verify https://github.com/nextcloud-releases/files_retention/releases/download/v2.0.0/files_retention-v2.0.0.tar.gz 8a99fbd808d9e561230ec706981f68a1f9649597 /tmp/files_retention.tgz
    tar xf /tmp/files_retention.tgz -C $CLOUD_DIR/apps/
    rm /tmp/files_retention.tgz

    # https://apps.nextcloud.com/apps/files_automatedtagging
    wget_verify https://github.com/nextcloud-releases/files_automatedtagging/releases/download/v2.0.0/files_automatedtagging-v2.0.0.tar.gz a532f0d6f81770029027fb6372fd8c30bb0eb064 /tmp/files_automatedtagging.tgz
    tar xf /tmp/files_automatedtagging.tgz -C $CLOUD_DIR/apps/
    rm /tmp/files_automatedtagging.tgz

    # https://apps.nextcloud.com/apps/groupfolders
    wget_verify https://github.com/nextcloud-releases/groupfolders/releases/download/v19.0.3/groupfolders-v19.0.3.tar.gz 08ac7404b17886589fe258e29d55a4213804aef1 /tmp/groupfolders.tgz
    tar xf /tmp/groupfolders.tgz -C $CLOUD_DIR/apps/
    rm /tmp/groupfolders.tgz

    # https://apps.nextcloud.com/apps/checksum
    wget_verify https://github.com/westberliner/checksum/releases/download/v1.2.5/checksum.tar.gz 808f28d315f1aa30543090dceeded22f8b951fa1 /tmp/checksum.tgz
    tar xf /tmp/checksum.tgz -C $CLOUD_DIR/apps/
    rm /tmp/checksum.tgz

    # https://apps.nextcloud.com/apps/fulltextsearch
    wget_verify https://github.com/nextcloud-releases/fulltextsearch/releases/download/31.0.0/fulltextsearch-31.0.0.tar.gz 0858ae675775c642efb8874b074f1158a7d239a3 /tmp/fulltextsearch.tgz
    tar xf /tmp/fulltextsearch.tgz -C $CLOUD_DIR/apps/
    rm /tmp/fulltextsearch.tgz

    # https://apps.nextcloud.com/apps/quota_warning
    wget_verify https://github.com/nextcloud-releases/quota_warning/releases/download/v1.21.0/quota_warning-v1.21.0.tar.gz a973fc59753dd124e9dd7ab763863650e90873ca /tmp/quota_warning.tgz
    tar xf /tmp/quota_warning.tgz -C $CLOUD_DIR/apps/
    rm /tmp/quota_warning.tgz

    # https://apps.nextcloud.com/apps/user_usage_report
    wget_verify https://github.com/nextcloud-releases/user_usage_report/releases/download/v2.0.0/user_usage_report-v2.0.0.tar.gz 1437f10a2d6cd8da7650054896e0ce57ab566cfb /tmp/user_usage_report.tgz
    tar xf /tmp/user_usage_report.tgz -C $CLOUD_DIR/apps/
    rm /tmp/user_usage_report.tgz

    # https://apps.nextcloud.com/apps/unsplash
    #wget_verify https://github.com/jancborchardt/unsplash/releases/download/v1.1.3/unsplash-1.1.3.tar.gz e4f8c62067cbf657fbff80f744b9991e55448520 /tmp/unsplash.tgz
    #tar xf /tmp/unsplash.tgz -C $CLOUD_DIR/apps/
    #rm /tmp/unsplash.tgz

    # https://apps.nextcloud.com/apps/twofactor_admin
    #wget_verify https://github.com/ChristophWurst/twofactor_admin/releases/download/v0.2.0/twofactor_admin.tar.gz c669e1c0c0323702d6a44d404f00a543cbb5d941  /tmp/twofactor_admin.tgz
    #tar xf /tmp/twofactor_admin.tgz -C $CLOUD_DIR/apps/
    #rm /tmp/twofactor_admin.tgz

    # https://apps.nextcloud.com/apps/contacts
    wget_verify https://github.com/nextcloud-releases/contacts/releases/download/v7.0.1/contacts-v7.0.1.tar.gz 9e4ae15e80e6df6444f62491eecf73ae63b12ab1 /tmp/contacts.tgz
    tar xf /tmp/contacts.tgz -C $CLOUD_DIR/apps/
    rm /tmp/contacts.tgz

    # https://apps.nextcloud.com/apps/calendar
    wget_verify https://github.com/nextcloud-releases/calendar/releases/download/v5.1.2/calendar-v5.1.2.tar.gz 4f5c310ebd1f65c66a4a5d9e70ea4701698d79ea /tmp/calendar.tgz
    tar xf /tmp/calendar.tgz -C $CLOUD_DIR/apps/
    rm /tmp/calendar.tgz

    # https://apps.nextcloud.com/apps/event_update_notification
    wget_verify https://github.com/nextcloud-releases/event_update_notification/releases/download/v2.6.1/event_update_notification-v2.6.1.tar.gz e1360a65c28df9670cf963830a9e836c8fb5016b /tmp/event_update_notification.tgz
    tar xf /tmp/event_update_notification.tgz -C $CLOUD_DIR/apps/
    rm /tmp/event_update_notification.tgz

    # https://apps.nextcloud.com/apps/notes
    wget_verify https://github.com/nextcloud-releases/notes/releases/download/v4.11.0/notes-v4.11.0.tar.gz 7f8f42387e05c9a2ecfc4c490861ebebcbaec264 /tmp/notes.tgz
    tar xf /tmp/notes.tgz -C $CLOUD_DIR/apps/
    rm /tmp/notes.tgz

    # https://apps.nextcloud.com/apps/tasks
    wget_verify https://github.com/nextcloud/tasks/releases/download/v0.16.1/tasks.tar.gz 69069c4bf0de8faf61a3d7b758d087e87cbeba27 /tmp/tasks.tgz
    tar xf /tmp/tasks.tgz -C $CLOUD_DIR/apps/
    rm /tmp/tasks.tgz

    # https://apps.nextcloud.com/apps/deck
    wget_verify https://github.com/nextcloud-releases/deck/releases/download/v1.15.0/deck-v1.15.0.tar.gz 92b15aa14f157045455d47dad2a3993141500f4e /tmp/deck.tgz
    tar xf /tmp/deck.tgz -C $CLOUD_DIR/apps/
    rm /tmp/deck.tgz

    # GPS apps
    # https://apps.nextcloud.com/apps/phonetrack
    wget_verify https://github.com/julien-nc/phonetrack/releases/download/v0.8.2/phonetrack-0.8.2.tar.gz 4f088acd122249e24cae05abdebcf0e3c72bdd9d /tmp/phonetrack.tgz
    tar xf /tmp/phonetrack.tgz -C $CLOUD_DIR/apps/
    rm /tmp/phonetrack.tgz

    # https://apps.nextcloud.com/apps/gpxpod
    wget_verify https://github.com/julien-nc/gpxpod/releases/download/v7.0.4/gpxpod-7.0.4.tar.gz 09d1bceab95bfb088ee28c61b82601e551e40a48 /tmp/gpxpod.tgz
    tar xf /tmp/gpxpod.tgz -C $CLOUD_DIR/apps/
    rm /tmp/gpxpod.tgz

    # Password apps
    # https://apps.nextcloud.com/apps/passman
    wget_verify https://releases.passman.cc/passman_2.4.11.tar.gz f7e55bc38958702007b655a95898f617c1c0eab6 /tmp/passman.tgz
    tar xf /tmp/passman.tgz -C $CLOUD_DIR/apps/
    rm /tmp/passman.tgz

    # https://apps.nextcloud.com/apps/keeweb
    wget_verify https://github.com/jhass/nextcloud-keeweb/releases/download/v0.6.20/keeweb-0.6.20.tar.gz 7980ab730d262e54e8e61ea3cfcd4ebde3846862 /tmp/keeweb.tgz
    tar xf /tmp/keeweb.tgz -C $CLOUD_DIR/apps/
    rm /tmp/keeweb.tgz

    # https://apps.nextcloud.com/apps/passwords
    wget_verify https://git.mdns.eu/api/v4/projects/45/packages/generic/passwords/2025.2.0/passwords.tar.gz 244c9efb6bf55fb922d06a448da5c9a20b43cd89 /tmp/passwords.tgz
    tar xf /tmp/keeweb.tgz -C $CLOUD_DIR/apps/
    rm /tmp/passwords.tgz

    # Social sharing apps
    # https://apps.nextcloud.com/apps/socialsharing_email
    wget_verify https://github.com/nextcloud-releases/socialsharing/releases/download/v3.3.0/socialsharing_email-v3.3.0.tar.gz b00832ac733662cf3883648406fa73da464ada4d /tmp/socialsharing_email.tgz
    tar xf /tmp/socialsharing_email.tgz -C $CLOUD_DIR/apps/
    rm /tmp/socialsharing_email.tgz

    # https://apps.nextcloud.com/apps/socialsharing_facebook
    wget_verify https://github.com/nextcloud-releases/socialsharing/releases/download/v3.3.0/socialsharing_facebook-v3.3.0.tar.gz dae509b53edb9e1a58a61b090b21e5deada5891c /tmp/socialsharing_facebook.tgz
    tar xf /tmp/socialsharing_facebook.tgz -C $CLOUD_DIR/apps/
    rm /tmp/socialsharing_facebook.tgz

    # https://apps.nextcloud.com/apps/socialsharing_twitter
    wget_verify https://github.com/nextcloud-releases/socialsharing/releases/download/v3.3.0/socialsharing_twitter-v3.3.0.tar.gz 0f357ab59e9148df28d9a12d0626818873c2575c /tmp/socialsharing_twitter.tgz
    tar xf /tmp/socialsharing_twitter.tgz -C $CLOUD_DIR/apps/
    rm /tmp/socialsharing_twitter.tgz

    # https://apps.nextcloud.com/apps/socialsharing_diaspora
    wget_verify https://github.com/nextcloud-releases/socialsharing/releases/download/v3.3.0/socialsharing_diaspora-v3.3.0.tar.gz cb366d346681daf95bd94dd105a4fab3bb601e9b /tmp/socialsharing_googleplus.tgz
    tar xf /tmp/socialsharing_googleplus.tgz -C $CLOUD_DIR/apps/
    rm /tmp/socialsharing_googleplus.tgz

    # https://apps.nextcloud.com/apps/socialsharing_bluesky
    wget_verify https://github.com/nextcloud-releases/socialsharing/releases/download/v3.3.0/socialsharing_bluesky-v3.3.0.tar.gz b4037e8cb26e658e3351a6d97310edc2120965b2 /tmp/socialsharing_bluesky.tgz
    tar xf /tmp/socialsharing_bluesky.tgz -C $CLOUD_DIR/apps/
    rm /tmp/socialsharing_bluesky.tgz
 
    # https://apps.nextcloud.com/apps/socialsharing_telegram
    wget_verify https://github.com/nextcloud-releases/socialsharing/releases/download/v3.3.0/socialsharing_telegram-v3.3.0.tar.gz b4ec7393cb98943fbffc1c57d834cff72c9afd8f /tmp/socialsharing_telegram.tgz
    tar xf /tmp/socialsharing_telegram.tgz -C $CLOUD_DIR/apps/
    rm /tmp/socialsharing_telegram.tgz

    # https://apps.nextcloud.com/apps/socialsharing_whatsapp
    wget_verify https://github.com/nextcloud-releases/socialsharing/releases/download/v3.3.0/socialsharing_whatsapp-v3.3.0.tar.gz b5e922c76dd9409ca8954b2855c22f3775b28d35 /tmp/socialsharing_whatsapp.tgz
    tar xf /tmp/socialsharing_whatsapp.tgz -C $CLOUD_DIR/apps/
    rm /tmp/socialsharing_whatsapp.tgz    
    
    # Bookmarks, Mail apps
    # https://apps.nextcloud.com/apps/bookmarks
    wget_spec_verify https://github.com/nextcloud/bookmarks/releases/download/v15.1.0/bookmarks-15.1.0.tar.gz 776e0c9a9679762d481428dc1a8928b9b7b3510c /tmp/bookmarks.tgz
    tar xf /tmp/bookmarks.tgz -C $CLOUD_DIR/apps/
    rm /tmp/bookmarks.tgz

    # https://apps.nextcloud.com/apps/mail
    wget_verify https://github.com/nextcloud-releases/mail/releases/download/v4.2.2/mail-stable4.2.tar.gz 2f93388a1d4d367316326fa5bdb7bb0e086e2c80 /tmp/mail.tgz
    tar xf /tmp/mail.tgz -C $CLOUD_DIR/apps/
    rm /tmp/mail.tgz

    # Video call
    # https://apps.nextcloud.com/apps/spreed
    wget_verify https://github.com/nextcloud-releases/spreed/releases/download/v21.0.0/spreed-v21.0.0.tar.gz 51ef52d47b9bee281a63f12aa755f97a2c650f12  /tmp/spreed.tgz
    tar xf /tmp/spreed.tgz -C $CLOUD_DIR/apps/
    rm /tmp/spreed.tgz
}
