#----------------------------------------------------------------------------
# Function: InstallWebmail
#    Install the chosen webmail client: Squirrelmail, Roundcube, RainLoop
#----------------------------------------------------------------------------
InstallWebmail() {
    local START_TIME=$SECONDS
    local VERSION=1.6.10
    local HASH=0cfbb457e230793df8c56c2e6d3655cf3818f168

    local UPDATE_KEY="$VERSION:$PERSISTENT_LOGIN_VERSION:$HTML5_NOTIFIER_VERSION:$FILTERS_VERSION:$RCM_RECAPTCHA:$CARDDAV_VERSION"

    echo -e "$IDENTATION_LVL_0 ${BWhite}Installing WebMail Client${NC}"

    if [[ "$CFG_WEBMAIL" == "roundcube" || "$CFG_WEBMAIL" == "roundcube-latest" ]]; then
        InstallRoundcube
    elif [[ "$CFG_WEBMAIL" == "squirrelmail" ]]; then
        InstallSquirrelmail
    elif [[ "$CFG_WEBMAIL" == "rainloop" ]]; then
        InstallRainLoop
    elif [[ "$CFG_WEBMAIL" == "sogo" ]]; then
        InstallSOGo
    fi

    # Enable PHP modules
    #phpenmod -v php8.2 mcrypt imap
    phpenmod -v php8.2 imap mbstring xml dom mcrypt intl
    systemctl restart php8.2-fpm
    
    # Stop Nginx to configure SSL
    systemctl stop nginx
    if [ "$CFG_SSL_NEW" == "yes" ]; then
	sh /home/$SUDO_USER/.acme.sh/acme.sh --issue --standalone -d webmail.$CFG_HOSTNAME_DOMAIN --force >> $PROGRAMS_INSTALL_LOG_FILES 2>&1
    else
	cp -av /home/$SUDO_USER/wtlinux_server_setup/acme_ssl/$CFG_HOSTNAME_DOMAIN/webmail.$CFG_HOSTNAME_DOMAIN /home/$SUDO_USER/.acme.sh/ >> $PROGRAMS_INSTALL_LOG_FILES 2>&1
    fi

    # Restart the appropriate web server
    if [[ "$CFG_WEBSERVER" == "apache" ]]; then
        service apache2 restart >> $PROGRAMS_INSTALL_LOG_FILES 2>&1
    else
        service nginx restart >> $PROGRAMS_INSTALL_LOG_FILES 2>&1
    fi

    # Log configuration details
    cat >> /home/administrator/wtlinux_server_setup/wtserver_configuration.txt <<EOF
# $CFG_WEBMAIL Store the passwords
#--------------------------------------------------------
Webmail software        : $CFG_WEBMAIL
Database type           : $CFG_SQLSERVER
Database name           : $CFG_ROUNDCUBE_DB
Database user           : $CFG_ROUNDCUBE_USR
Database password       : $CFG_ROUNDCUBE_PWD

Ispconfig3 soap user    : $CFG_ROUNDCUBE_SOAP_USR
Ispconfig3 soap password: $CFG_ROUNDCUBE_SOAP_PWD
#--------------------------------------------------------
EOF

    MeasureTimeDuration $START_TIME
}

#---------------------------------------------------------------------
# Function: InstallRoundcube
#    Install and configure Roundcube WebMail
#---------------------------------------------------------------------
InstallRoundcube() {
    # paths that are often reused
    local RCM_DIR=/usr/share/roundcube
    local RCM_PLUGIN_DIR=${RCM_DIR}/plugins
    local RCM_CONFIG=${RCM_DIR}/config/config.inc.php
    
    echo -e "$IDENTATION_LVL_1 Installing ${red}RoundCube${NC} WebMail Client..."

    if [[ "$CFG_WEBMAIL" == "roundcube-latest" ]]; then
        echo -e "$IDENTATION_LVL_2 Installing the latest version of Roundcube from source (version $VERSION)..."
        InstallRoundcubeLatest
    else
        echo -e "$IDENTATION_LVL_2 Installing Roundcube from the Debian repository (stable version)..."
        InstallRoundcubeStable
    fi

    echo -e "$IDENTATION_LVL_2 Roundcube installation and configuration completed. [${green}DONE${NC}]"
}

#---------------------------------------------------------------------
# Function: InstallRoundcubeStable
#    Install and configure the stable version of Roundcube
#---------------------------------------------------------------------
InstallRoundcubeStable() {
    # Configure Roundcube database
    echo -e "$IDENTATION_LVL_2 Configuring Roundcube database... "
    #CFG_ROUNDCUBE_PWD=$(< /dev/urandom tr -dc A-Z-a-z-0-9 | head -c16)
    echo "roundcube-core roundcube/dbconfig-install boolean true" | debconf-set-selections
    echo "roundcube-core roundcube/database-type select mysql" | debconf-set-selections
    echo "roundcube-core roundcube/mysql/admin-pass password $CFG_MYSQL_ROOT_PWD" | debconf-set-selections
    echo "roundcube-core roundcube/db/dbname string roundcube" | debconf-set-selections
    echo "roundcube-core roundcube/mysql/app-pass password $CFG_ROUNDCUBE_PWD" | debconf-set-selections
    echo "roundcube-core roundcube/app-password-confirm password $CFG_ROUNDCUBE_PWD" | debconf-set-selections
    echo "roundcube-core roundcube/hosts string localhost" | debconf-set-selections
    echo -e "[${green}DONE${NC}]"

    # Install Roundcube packages
    echo -e "$IDENTATION_LVL_2 Installing Roundcube packages... "
    apt_install roundcube-core roundcube roundcube-mysql roundcube-plugins
    echo -e "[${green}DONE${NC}]"

    # Configure Roundcube on WebServer
    echo -e "$IDENTATION_LVL_2 Configuring Roundcube on WebServer... "
    ConfigureRoundcubeWebServer
    echo -e "[${green}DONE${NC}]"

    # Integrate Roundcube with ISPConfig
    echo -e "$IDENTATION_LVL_2 Integrating Roundcube with ISPConfig... "
    IntegrateRoundcubeWithISPConfig
    echo -e "[${green}DONE${NC}]"
}

#---------------------------------------------------------------------
# Function: InstallRoundcubeLatest
#    Install and configure the latest version of Roundcube
#---------------------------------------------------------------------
InstallRoundcubeLatest() {
    #if [ "$CFG_MULTISERVER" == "no" ] && [ "$CFG_MIRRORSERVER" == "no" ]; then
	# Generate a safe 24-character secret key of safe characters.
	#CFG_ROUNDCUBE_PWD=$(< /dev/urandom tr -dc A-Z-a-z-0-9 | head -c12)
	#deskey=$(cat /dev/urandom | tr -dc 'a-zA-Z0-9-_#&!*%?' | fold -w 24 | head -n 1)
	##CFG_ROUNDCUBE_PWD=$(dd if=/dev/urandom bs=1 count=18 2>/dev/null | base64 | fold -w 24 | head -n 1)
	##CFG_ROUNDCUBE_DB="roundcube"
	##CFG_ROUNDCUBE_USR="roundcube"
    #fi
	    
    # Install required PHP packages
    echo -e "$IDENTATION_LVL_2 Installing required PHP packages... "
    apt_install -t php8.2 php8.2-gd php8.2-mysql php8.2-curl php8.2-zip php8.2-ldap php8.2-mbstring php8.2-imagick php8.2-intl php8.2-xml php8.2-dom php8.2-xmlreader php8.2-xmlwriter php8.2-gettext php8.2-json php8.2-gd php8.2-bcmath php8.2-cgi php8.2-gmp php8.2-common php8.2-fpm php-pear unzip wget curl javascript-common libjs-jquery-mousewheel libjs-jquery libmagic1 tinymce
    echo -e "[${green}DONE${NC}]"

    echo -e "$IDENTATION_LVL_2 Installing required PHP pear packages... "
    pear channel-update pear.php.net
    pear install Mail_Mime
    pear install Auth_SASL
    pear install Net_SMTP
    pear install Net_IDNA2
    echo -e "[${green}DONE${NC}]"
    
    apt_purge_remove roundcube*

    needs_update=0
    if [ ! -f /usr/local/lib/roundcubemail/version ]; then
	# not installed yet
	needs_update=1
    elif [[ "$UPDATE_KEY" != `cat /usr/local/lib/roundcubemail/version` ]]; then
	# checks if the version is what we want
	needs_update=1
    fi
    if [ $needs_update == 1 ]; then
	# Install Roundcube from source
	echo -e "$IDENTATION_LVL_2 Installing Roundcube from source... "
	wget_verify \
	    https://github.com/roundcube/roundcubemail/releases/download/$VERSION/roundcubemail-$VERSION-complete.tar.gz \
	    $HASH \
	    /tmp/roundcube.tgz

	tar -C /usr/local/lib --no-same-owner -zxf /tmp/roundcube.tgz
	rm -rf /usr/local/lib/roundcubemail
	mv /usr/local/lib/roundcubemail-$VERSION/ $RCM_DIR
	rm -f /tmp/roundcube.tgz
	echo -e "[${green}DONE${NC}]"
    fi
    # Configure Roundcube database
    echo -e "$IDENTATION_LVL_2 Configuring Roundcube database... "
    mariadb -uroot -p$CFG_MYSQL_ROOT_PWD -e "CREATE DATABASE ${CFG_ROUNDCUBE_DB}" > /dev/null 2>&1
    mariadb -uroot -p$CFG_MYSQL_ROOT_PWD -e "GRANT ALL PRIVILEGES ON ${CFG_ROUNDCUBE_DB}.* TO ${CFG_ROUNDCUBE_USR}@localhost IDENTIFIED BY '$CFG_ROUNDCUBE_PWD'" > /dev/null 2>&1
    mariadb -uroot -p$CFG_MYSQL_ROOT_PWD -e "flush privileges;" > /dev/null 2>&1
    mariadb -uroot -p"$CFG_MYSQL_ROOT_PWD" ${CFG_ROUNDCUBE_DB} < /usr/share/roundcube/SQL/mysql.initial.sql
    cp -pf /usr/share/roundcube/config/config.inc.php.sample /usr/share/roundcube/config/config.inc.php
    CFG_ROUNDCUBE_PRODUCTNAME=$(echo "$CFG_HOSTNAME_DOMAIN" | cut -d. -f1 | sed 's/\b\(.\)/\u\1/g')
    sed -i "s/\$config\['product_name'\] = 'Roundcube Webmail';/\$config\['product_name'\] = '$CFG_ROUNDCUBE_PRODUCTNAME Webmail';/" /usr/share/roundcube/config/config.inc.php

    # Create writable directories
    mkdir -p /var/log/roundcubemail /tmp/roundcubemail
    chown -R www-data.www-data /var/log/roundcubemail /tmp/roundcubemail
    
    # Ensure the log file monitored by fail2ban exists, or else fail2ban can't start.
    sudo -u www-data touch /var/log/roundcubemail/errors
    echo -e "[${green}DONE${NC}]"

    # Install Roundcube plugins
    echo -e "$IDENTATION_LVL_2 Installing Roundcube plugins... "
    InstallRoundcubePlugins
    echo -e "[${green}DONE${NC}]"

    # Configure Roundcube on WebServer
    echo -e "$IDENTATION_LVL_2 Configuring Roundcube on WebServer... "
    ConfigureRoundcubeWebServer
    echo -e "[${green}DONE${NC}]"

    # Integrate Roundcube with ISPConfig
    echo -e "$IDENTATION_LVL_2 Integrating Roundcube with ISPConfig... "
    IntegrateRoundcubeWithISPConfig
    echo -e "[${green}DONE${NC}]"
}

#---------------------------------------------------------------------
# Function: InstallRoundcubePlugins
#    Install Roundcube plugins
#---------------------------------------------------------------------
InstallRoundcubePlugins() {
    local PERSISTENT_LOGIN_VERSION=bde7b6840c7d91de627ea14e81cf4133cbb3c07a
    local HTML5_NOTIFIER_VERSION=5d85947e16ea9cac4544c59666fe0ff13ba2a44a
    local FILTERS_VERSION=231e19b4aca1af5baf94c59ef732298c3d2c1428
    local LOGINSEC_VERSION=5db3d4808477557260fcf0af9e30741379920759
    local RCM_RECAPTCHA=f1d89d0a3fc954ba595081494b7c664eb6dc1ea7
    local CARDDAV_VERSION=5.1.0
    local CARDDAV_HASH=76a92defface4ac136eae7ea5e73951f73025bf1
    local TWO_FACTOR_VERSION=7681a3744fda1c4858a10cc000e4995df7df41de

    # Install Persistent Login plugin
    git_clone https://github.com/mfreiholz/Roundcube-Persistent-Login-Plugin.git $PERSISTENT_LOGIN_VERSION '' ${RCM_PLUGIN_DIR}/persistent_login

    # Install HTML5 Notifier plugin
    git_clone https://github.com/kitist/html5_notifier.git $HTML5_NOTIFIER_VERSION '' ${RCM_PLUGIN_DIR}/html5_notifier

    # Install Filters plugin
    git_clone https://github.com/yoshioki/roundcube-plugin-filters $FILTERS_VERSION '' ${RCM_PLUGIN_DIR}/filters

    # Install reCAPTCHA plugin
    git_clone https://github.com/sjinks/roundcube_recaptcha $RCM_RECAPTCHA '' ${RCM_PLUGIN_DIR}/roundcube_recaptcha
    
    # Install Login Security plugin
    git_clone https://github.com/dapphp/Roundcube-Plugin-login_security $LOGINSEC_VERSION '' ${RCM_PLUGIN_DIR}/login_security

    # Install Two-Factor Authentication plugin
    git_clone https://github.com/alexandregz/twofactor_gauthenticator $TWO_FACTOR_VERSION '' ${RCM_PLUGIN_DIR}/twofactor_gauthenticator
    
    # Install CardDAV plugin
    git_clone https://github.com/mstilkerich/rcmcarddav $CARDDAV_HASH '' ${RCM_PLUGIN_DIR}/carddav

    # Configure CardDAV plugin
    echo -e "
<?php
/* Do not edit. Written by Wolftronics. Regenerated on updates. */
\$prefs['_GLOBAL']['hide_preferences'] = true;
\$prefs['_GLOBAL']['suppress_version_warning'] = true;
\$prefs['ownCloud'] = array(
    'name'         =>  'ownCloud',
    'username'     =>  '%u', // login username
    'password'     =>  '%p', // login password
    'url'          =>  'https://cloud.$CFG_HOSTNAME_DOMAIN/cloud/remote.php/carddav/addressbooks/%u/contacts',
    'active'       =>  true,
    'readonly'     =>  false,
    'refresh_time' => '02:00:00',
    'fixed'        =>  array('username','password'),
    'preemptive_auth' => '1',
    'hide'        =>  false,
);" > ${RCM_PLUGIN_DIR}/carddav/config.inc.php

    # Record the installed version
    echo $UPDATE_KEY > ${RCM_DIR}/version
}

# Function: ConfigureRoundcubeWebServer
#    Configure Roundcube on the web server (Apache or Nginx)
#---------------------------------------------------------------------
ConfigureRoundcubeWebServer() {
    ln -s /usr/share/roundcube /var/lib/roundcube
    if [[ "$CFG_WEBSERVER" == "apache" ]]; then
        ConfigureApacheForRoundcube
    elif [[ "$CFG_WEBSERVER" == "nginx" ]]; then
        ConfigureNginxForRoundcube
    fi
}

#---------------------------------------------------------------------
# Function: ConfigureApacheForRoundcube
#    Configure Apache for Roundcube
#---------------------------------------------------------------------
ConfigureApacheForRoundcube() {
    echo -e "$IDENTATION_LVL_2 Configuring Apache for Roundcube..."

    # Backup the original Apache configuration file
    mv /etc/roundcube/apache.conf /etc/roundcube/apache.conf.default

    # Create a new Apache configuration for Roundcube
    cat > /etc/roundcube/apache.conf <<EOF
<VirtualHost *:80>
    Alias /webmail /var/lib/roundcube
    <Directory /var/lib/roundcube/>
        Options +FollowSymLinks
	# This is needed to parse /var/lib/roundcube/.htaccess. See its
	# content before setting AllowOverride to None.
        AllowOverride All
        <IfVersion >= 2.3>
            Require all granted
        </IfVersion>
        <IfVersion < 2.3>
            Order allow,deny
            Allow from all
        </IfVersion>
    </Directory>

    # Protecting basic directories:
    <Directory /var/lib/roundcube/config>
        Options -FollowSymLinks
        AllowOverride None
    </Directory>

    <Directory /var/lib/roundcube/temp>
        Options -FollowSymLinks
        AllowOverride None
        <IfVersion >= 2.3>
            Require all denied
        </IfVersion>
        <IfVersion < 2.3>
            Order allow,deny
            Deny from all
        </IfVersion>
    </Directory>

    <Directory /var/lib/roundcube/logs>
        Options -FollowSymLinks
        AllowOverride None
        <IfVersion >= 2.3>
            Require all denied
        </IfVersion>
        <IfVersion < 2.3>
            Order allow,deny
            Deny from all
        </IfVersion>
    </Directory>
</VirtualHost>

<IfModule mod_ssl.c>
<VirtualHost *:443>
    Alias /webmail /var/lib/roundcube
    <Directory /var/lib/roundcube/>
        Options +FollowSymLinks
 	# This is needed to parse /var/lib/roundcube/.htaccess. See its
	# content before setting AllowOverride to None.       
        AllowOverride All
        <IfVersion >= 2.3>
            Require all granted
        </IfVersion>
        <IfVersion < 2.3>
            Order allow,deny
            Allow from all
        </IfVersion>
    </Directory>

    # Protecting basic directories:
    <Directory /var/lib/roundcube/config>
        Options -FollowSymLinks
        AllowOverride None
    </Directory>

    <Directory /var/lib/roundcube/temp>
        Options -FollowSymLinks
        AllowOverride None
        <IfVersion >= 2.3>
            Require all denied
        </IfVersion>
        <IfVersion < 2.3>
            Order allow,deny
            Deny from all
        </IfVersion>
    </Directory>

    <Directory /var/lib/roundcube/logs>
        Options -FollowSymLinks
        AllowOverride None
        <IfVersion >= 2.3>
            Require all denied
        </IfVersion>
        <IfVersion < 2.3>
            Order allow,deny
            Deny from all
        </IfVersion>
    </Directory>

    # SSL Configuration
    SSLEngine On
    SSLProtocol All -SSLv2 -SSLv3
    #SSLCertificateFile    /usr/local/ispconfig/interface/ssl/ispserver.crt
    #SSLCertificateKeyFile /usr/local/ispconfig/interface/ssl/ispserver.key
    SSLCertificateFile     /home/$SUDO_USER/.acme.sh/webmail.$CFG_HOSTNAME_DOMAIN/fullchain.cer
    SSLCertificateKeyFile  /home/$SUDO_USER/.acme.sh/webmail.$CFG_HOSTNAME_DOMAIN/webmail.$CFG_HOSTNAME_DOMAIN.key
    SSLCipherSuite ECDHE-ECDSA-AES256-GCM-SHA384:ECDHE-RSA-AES256-GCM-SHA384:ECDHE-ECDSA-CHACHA20-POLY1305:ECDHE-RSA-CHACHA20-POLY1305:ECDHE-ECDSA-AES128-GCM-SHA256:ECDHE-RSA-AES128-GCM-SHA256:DHE-RSA-AES256-GCM-SHA384:DHE-RSA-AES128-GCM-SHA256
    SSLHonorCipherOrder on
</VirtualHost>
</IfModule>
EOF

    echo -e "$IDENTATION_LVL_2 Apache configuration for Roundcube completed. [${green}DONE${NC}]"
}

#---------------------------------------------------------------------
# Function: ConfigureNginxForRoundcube
#    Configure Nginx for Roundcube
#---------------------------------------------------------------------
ConfigureNginxForRoundcube() {
    echo -e "$IDENTATION_LVL_2 Configuring Nginx for Roundcube..."

    # Determine the PHP socket path based on the installed PHP version
    #local APPS_SOCKET="/var/run/php/php8.2-fpm.sock"
    local APPS_SOCKET=$(GetPHPSocketPath)

    # Create a new Nginx configuration for Roundcube
    cat > /etc/nginx/sites-available/webmail-roundcube.vhost <<EOF
server {
    listen 80;
    server_name webmail.$CFG_HOSTNAME_DOMAIN;
    return 301 https://\$server_name\$request_uri;
}

server {
    #listen 80 http2;
    listen 443 ssl http2;
    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_certificate        /home/$SUDO_USER/.acme.sh/webmail.$CFG_HOSTNAME_DOMAIN/fullchain.cer;
    ssl_certificate_key    /home/$SUDO_USER/.acme.sh/webmail.$CFG_HOSTNAME_DOMAIN/webmail.$CFG_HOSTNAME_DOMAIN.key;

    ssl_ciphers 'ECDHE-ECDSA-AES256-GCM-SHA384:ECDHE-RSA-AES256-GCM-SHA384:ECDHE-ECDSA-CHACHA20-POLY1305:ECDHE-RSA-CHACHA20-POLY1305:ECDHE-ECDSA-AES128-GCM-SHA256:ECDHE-RSA-AES128-GCM-SHA256:DHE-RSA-AES256-GCM-SHA384:DHE-RSA-AES128-GCM-SHA256';
    ssl_prefer_server_ciphers on;
    ssl_session_timeout 1d;
    ssl_session_cache shared:SSL:50m;
    ssl_session_tickets off;

    # OCSP Stapling
    ssl_stapling on;
    ssl_stapling_verify on;
    resolver 8.8.8.8 8.8.4.4 valid=300s;
    resolver_timeout 5s;

    # HSTS configuration
    add_header Strict-Transport-Security "max-age=63072000" always;

    # If you want browsers to know before the first visit that the website is only
    # available via HTTPS, you can register the domain on the HSTS Preload list https://hstspreload.org/.
    #add_header Strict-Transport-Security "max-age=63072000; includeSubDomains; preload" always;

    server_name webmail.*;
    root /var/lib/roundcube;
    index index.php index.html;

    location ~ ^/favicon.ico$ {
        root /var/lib/roundcube/skins/default/images;
        log_not_found off;
        access_log off;
        expires max;
    }

    location = /robots.txt {
        allow all;
        log_not_found off;
        access_log off;
    }

    location ~ ^/(README|INSTALL|LICENSE|CHANGELOG|UPGRADING)$ {
        deny all;
    }

    location ~ ^/(bin|SQL)/ {
        deny all;
    }

    location ~* /.svn/ {
        deny all;
    }

    location ~ /\. {
        deny all;
        access_log off;
        log_not_found off;
    }

    location ~ \.php$ {
        include /etc/nginx/fastcgi_params;
        fastcgi_param   REDIRECT_STATUS         200;
        #fastcgi_pass unix:/var/run/php/php8.2-fpm.sock;
        fastcgi_pass unix:$APPS_SOCKET;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME \$document_root\$fastcgi_script_name;
        fastcgi_buffer_size 128k;
        fastcgi_buffers 256 4k;
        fastcgi_busy_buffers_size 256k;
        fastcgi_temp_file_write_size 256k;
    }
}
EOF

    # Enable the Nginx configuration
    ln -s /etc/nginx/sites-available/webmail-roundcube.vhost /etc/nginx/sites-enabled/000-webmail-roundcube.vhost

    echo -e "$IDENTATION_LVL_2 Nginx configuration for Roundcube completed. [${green}DONE${NC}]"
}

#---------------------------------------------------------------------
# Function: IntegrateRoundcubeWithISPConfig
#    Integrate Roundcube with ISPConfig
#---------------------------------------------------------------------
IntegrateRoundcubeWithISPConfig() {
    wget -q --no-check-certificate -O $PROGRAMS_INSTALL_DOWNLOAD/ispconfig3_roundcube.tgz https://github.com/w2c/ispconfig3_roundcube/tarball/master
    mkdir -p $PROGRAMS_INSTALL_DOWNLOAD/ispconfig3_roundcube && tar xzf $PROGRAMS_INSTALL_DOWNLOAD/ispconfig3_roundcube.tgz -C $PROGRAMS_INSTALL_DOWNLOAD/ispconfig3_roundcube
    cp -r $PROGRAMS_INSTALL_DOWNLOAD/ispconfig3_roundcube/*ispconfig3_roundcube*/ispconfig3_* /usr/share/roundcube/plugins/

    for plugin in account autoreply autoselect fetchmail filter forward pass spam wblist; do
        ln -s /usr/share/roundcube/plugins/ispconfig3_$plugin /var/lib/roundcube/plugins/ispconfig3_$plugin
    done
	    
    sed -i "s|^\(\$config\['default_host'\] =\).*$|\1 \ array(\n'mail.${CFG_HOSTNAME_DOMAIN}' => 'Default Server',\n'webmail.${CFG_HOSTNAME_FQDN}.hu' => 'Webmail Server',\n'ssl://mail.${CFG_HOSTNAME_DOMAIN}:993' => 'Secure Webmail Server'\n\);|" /usr/share/roundcube/config/config.inc.php
    sed -i "s|^\(\$config\['db_dsnw'\] =\).*$|\1 \'mysqli://${CFG_ROUNDCUBE_DB}:${CFG_ROUNDCUBE_PWD}@localhost/${CFG_ROUNDCUBE_DB}\';|" /usr/share/roundcube/config/config.inc.php
    sed -i "s|^\(\$config\['smtp_host'\] =\).*$|\1 \'%h\';|" /usr/share/roundcube/config/config.inc.php
    #sed -i "s|^\(\$config\['smtp_port'\] =\).*$|\1 \'587\';|" /usr/share/roundcube/config/config.inc.php
    sed -i "s|^\(\$config\['smtp_user'\] =\).*$|\1 \'%u\';|" /usr/share/roundcube/config/config.inc.php
    sed -i "s|^\(\$config\['smtp_pass'\] =\).*$|\1 \'%p\';|" /usr/share/roundcube/config/config.inc.php
    sed -i "s|^\(\$config\['product_name'\] =\).*$|\1 \'${CFG_ROUNDCUBE_PRODUCTNAME}\';|" /usr/share/roundcube/config/config.inc.php
    #sed -i "s|^\(\$config\['support_url'\] =\).*$|\1 \'mailto:${E}\';|" /usr/share/roundcube/config/config.inc.php
    #sed -i "s|^\(\$config\['des_key'\] =\).*$|\1 \'${deskey}\';|" /usr/share/roundcube/config/config.inc.php

    sed -i "/'zipdownload',/a 'jqueryui',\n'filters',\n'html5_notifier',\n'persistent_login',\n'managesieve',\n'emoticons',\n'ispconfig3_account',\n'ispconfig3_autoreply',\n'ispconfig3_autoselect',\n'ispconfig3_pass',\n'ispconfig3_spam',\n'ispconfig3_fetchmail',\n'ispconfig3_filter',\n'ispconfig3_forward',\n'ispconfig3_wblist',\n'login_security',\n'twofactor_gauthenticator'," /usr/share/roundcube/config/config.inc.php
    
    echo "\$config['login_security_max_failed_logins'] = 3;" >> /usr/share/roundcube/config/config.inc.php
    echo "\$config['login_security_block_time'] = 600;" >> /usr/share/roundcube/config/config.inc.php
    echo "\$config['twofactor_gauthenticator'] = array('enabled' => true, 'issuer' => '${CFG_ROUNDCUBE_PRODUCTNAME}');" >> /usr/share/roundcube/config/config.inc.php
    
    echo "\$config['login_autocomplete'] = 2;" >> /usr/share/roundcube/config/config.inc.php
    echo "\$config['password_charset'] = 'UTF-8';" >> /usr/share/roundcube/config/config.inc.php
    echo "\$config['junk_mbox'] = 'Spam';" >> /usr/share/roundcube/config/config.inc.php
    echo "\$config['log_dir'] = '/var/log/roundcubemail/';" >> /usr/share/roundcube/config/config.inc.php
    echo "\$config['temp_dir'] = '/tmp/roundcubemail/';" >> /usr/share/roundcube/config/config.inc.php
	    
    mv /usr/share/roundcube/plugins/ispconfig3_account/config/config.inc.php.dist /usr/share/roundcube/plugins/ispconfig3_account/config/config.inc.php
    sed -i "s/\(\$config\['remote_soap_user'\] = \)'.*';/\1'$CFG_ROUNDCUBE_SOAP_USR';/g" /usr/share/roundcube/plugins/ispconfig3_account/config/config.inc.php
    sed -i "s/\(\$config\['remote_soap_pass'\] = \)'.*';/\1'$CFG_ROUNDCUBE_SOAP_PWD';/g" /usr/share/roundcube/plugins/ispconfig3_account/config/config.inc.php
    sed -i "s/\(\$config\['soap_url'\] = \)'.*';/\1'https\:\/\/$IP_ADDRESS\:$CFG_ISPONCFIG_PORT\/remote\/';/g" /usr/share/roundcube/plugins/ispconfig3_account/config/config.inc.php

    mv /usr/share/roundcube/plugins/ispconfig3_pass/config/config.inc.php.dist /usr/share/roundcube/plugins/ispconfig3_pass/config/config.inc.php
    sed -i "s/\$config\['password_min_length'\] = 6;/\$config\['password_min_length'\] = 8;/" /usr/share/roundcube/plugins/ispconfig3_pass/config/config.inc.php
    sed -i "s/\$config\['password_check_symbol'\] = TRUE;/\$config\['password_check_symbol'\] = FALSE;/" /usr/share/roundcube/plugins/ispconfig3_pass/config/config.inc.php
    rm -rf /usr/share/roundcube/installer

    # Fix Carddav permissions:
    chown -R root.www-data ${RCM_PLUGIN_DIR}/carddav
    # root.www-data need all permissions, others only read
    chmod -R 774 ${RCM_PLUGIN_DIR}/carddav

    ${RCM_DIR}/bin/updatedb.sh --dir ${RCM_DIR}/SQL --package roundcube
    composer require roundcube/larry
}

#---------------------------------------------------------------------
# Function: InstallRainLoop
#    Install and configure RainLoop WebMail
#---------------------------------------------------------------------
InstallRainLoop() {
    echo -e "$IDENTATION_LVL_1 Installing ${red}RainLoop${NC} WebMail Client..."

    # Configure Roundcube database
    echo -e "$IDENTATION_LVL_2 Configuring RainLoop database... "
    mysql -uroot -p$CFG_MYSQL_ROOT_PWD -e "CREATE DATABASE ${CFG_ROUNDCUBE_DB}" > /dev/null 2>&1
    mysql -uroot -p$CFG_MYSQL_ROOT_PWD -e "CREATE USER '$CFG_ROUNDCUBE_USR'@'localhost' IDENTIFIED BY '$CFG_ROUNDCUBE_PWD'" > /dev/null 2>&1
    mysql -uroot -p$CFG_MYSQL_ROOT_PWD -e "GRANT ALL PRIVILEGES ON ${CFG_ROUNDCUBE_DB}.* TO ${CFG_ROUNDCUBE_USR}@localhost IDENTIFIED BY '$CFG_ROUNDCUBE_PWD'" > /dev/null 2>&1
    mysql -uroot -p$CFG_MYSQL_ROOT_PWD -e "flush privileges;" > /dev/null 2>&1
 
    # Download and install RainLoop
    mkdir -p /var/www/rainloop
    wget -q -O /tmp/rainloop-latest.zip https://www.rainloop.net/repository/webmail/rainloop-latest.zip
    unzip -q /tmp/rainloop-latest.zip -d /var/www/rainloop
    rm -f /tmp/rainloop-latest.zip

    # Set permissions
    chown -R www-data:www-data /var/www/rainloop
    find /var/www/rainloop -type d -exec chmod 755 {} \;
    find /var/www/rainloop -type f -exec chmod 644 {} \;

    # Configure RainLoop for Apache or Nginx
    if [[ "$CFG_WEBSERVER" == "apache" ]]; then
        ConfigureApacheForRainLoop
    elif [[ "$CFG_WEBSERVER" == "nginx" ]]; then
        ConfigureNginxForRainLoop
    fi

    echo -e "$IDENTATION_LVL_2 RainLoop installation and configuration completed. [${green}DONE${NC}]"
}

#---------------------------------------------------------------------
# Function: ConfigureApacheForRainLoop
#    Configure Apache for RainLoop
#---------------------------------------------------------------------
ConfigureApacheForRainLoop() {
    echo -e "$IDENTATION_LVL_2 Configuring Apache for RainLoop..."

    # Create Apache configuration for RainLoop
    cat > /etc/apache2/sites-available/rainloop.conf <<EOF
<VirtualHost *:80>
    DocumentRoot /var/www/rainloop
    ServerName webmail.$CFG_HOSTNAME_DOMAIN

    <Directory /var/www/rainloop>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>

    ErrorLog \${APACHE_LOG_DIR}/rainloop-error.log
    CustomLog \${APACHE_LOG_DIR}/rainloop-access.log combined
</VirtualHost>

<IfModule mod_ssl.c>
<VirtualHost *:443>
    DocumentRoot /var/www/rainloop
    ServerName webmail.$CFG_HOSTNAME_DOMAIN

    <Directory /var/www/rainloop>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>

    ErrorLog \${APACHE_LOG_DIR}/rainloop-error.log
    CustomLog \${APACHE_LOG_DIR}/rainloop-access.log combined

    SSLEngine on
    SSLCertificateFile /home/$SUDO_USER/.acme.sh/webmail.$CFG_HOSTNAME_DOMAIN/fullchain.cer
    SSLCertificateKeyFile /home/$SUDO_USER/.acme.sh/webmail.$CFG_HOSTNAME_DOMAIN/webmail.$CFG_HOSTNAME_DOMAIN.key
    SSLCipherSuite ECDHE-ECDSA-AES256-GCM-SHA384:ECDHE-RSA-AES256-GCM-SHA384:ECDHE-ECDSA-CHACHA20-POLY1305:ECDHE-RSA-CHACHA20-POLY1305:ECDHE-ECDSA-AES128-GCM-SHA256:ECDHE-RSA-AES128-GCM-SHA256:DHE-RSA-AES256-GCM-SHA384:DHE-RSA-AES128-GCM-SHA256
    SSLHonorCipherOrder on
</VirtualHost>
</IfModule>
EOF

    # Enable the site and restart Apache
    a2ensite rainloop.conf
    systemctl restart apache2

    echo -e "$IDENTATION_LVL_2 Apache configuration for RainLoop completed. [${green}DONE${NC}]"
}

#---------------------------------------------------------------------
# Function: ConfigureNginxForRainLoop
#    Configure Nginx for RainLoop
#---------------------------------------------------------------------
ConfigureNginxForRainLoop() {
    echo -e "$IDENTATION_LVL_2 Configuring Nginx for RainLoop..."

    # Determine the PHP socket path based on the installed PHP version
    local APPS_SOCKET=$(GetPHPSocketPath)

    # Create Nginx configuration for RainLoop
    cat > /etc/nginx/sites-available/rainloop.conf <<EOF
server {
    listen 80;
    server_name webmail.$CFG_HOSTNAME_DOMAIN;
    return 301 https://\$server_name\$request_uri;
}

server {
    listen 443 ssl http2;
    server_name webmail.$CFG_HOSTNAME_DOMAIN;

    root /var/www/rainloop;
    index index.php;

    ssl_certificate /home/$SUDO_USER/.acme.sh/webmail.$CFG_HOSTNAME_DOMAIN/fullchain.cer;
    ssl_certificate_key /home/$SUDO_USER.acme.sh/webmail.$CFG_HOSTNAME_DOMAIN/webmail.$CFG_HOSTNAME_DOMAIN.key;
    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_ciphers 'ECDHE-ECDSA-AES256-GCM-SHA384:ECDHE-RSA-AES256-GCM-SHA384:ECDHE-ECDSA-CHACHA20-POLY1305:ECDHE-RSA-CHACHA20-POLY1305:ECDHE-ECDSA-AES128-GCM-SHA256:ECDHE-RSA-AES128-GCM-SHA256:DHE-RSA-AES256-GCM-SHA384:DHE-RSA-AES128-GCM-SHA256';
    ssl_prefer_server_ciphers on;
    ssl_session_timeout 1d;
    ssl_session_cache shared:SSL:50m;
    ssl_session_tickets off;

    # HSTS configuration
    add_header Strict-Transport-Security "max-age=63072000" always;

    location / {
        try_files \$uri \$uri/ /index.php?\$query_string;
    }

    location ~ \.php$ {
        fastcgi_index index.php;
        fastcgi_split_path_info ^(.+\.php)(.*)$;
        fastcgi_keep_conn on;    
        include snippets/fastcgi-php.conf;
        fastcgi_pass unix:$APPS_SOCKET;
        fastcgi_param SCRIPT_FILENAME \$document_root\$fastcgi_script_name;
        include fastcgi_params;
    }
        
    location ~ /\.ht {
        deny all;
    }

    location ~* ^/(data|config|\.env|\.htaccess|\.user.ini|\.git|\.svn|\.DS_Store) {
        deny all;
    }
}
EOF

    # Enable the site and restart Nginx
    ln -s /etc/nginx/sites-available/rainloop.conf /etc/nginx/sites-enabled/
    systemctl restart nginx

    echo -e "$IDENTATION_LVL_2 Nginx configuration for RainLoop completed. [${green}DONE${NC}]"
}

#---------------------------------------------------------------------
# Function: InstallSOGo
#    Install and configure SOGo Groupware from GitHub
#---------------------------------------------------------------------
InstallSOGo() {
    echo -e "$IDENTATION_LVL_1 Installing ${red}SOGo${NC} Groupware from GitHub..."

    # Install dependencies
    apt_install gnustep-make gnustep-base-runtime gnustep-base-common gnustep-base-dev \
                libgnustep-base1.27 libgnustep-base-dev gobjc build-essential

    # Clone SOGo from GitHub
    local SOGO_DIR=/usr/local/src/sogo
    mkdir -p $SOGO_DIR
    git clone https://github.com/inverse-inc/sogo.git $SOGO_DIR
    cd $SOGO_DIR

    # Build and install SOGo
    ./configure --prefix=/usr/local --disable-debug --enable-release
    make
    make install

    # Create SOGo configuration directory
    mkdir -p /etc/sogo
    chown -R sogo:sogo /etc/sogo

    # Configure SOGo for Apache or Nginx
    if [[ "$CFG_WEBSERVER" == "apache" ]]; then
        ConfigureApacheForSOGo
    elif [[ "$CFG_WEBSERVER" == "nginx" ]]; then
        ConfigureNginxForSOGo
    fi

    echo -e "$IDENTATION_LVL_2 SOGo installation and configuration completed. [${green}DONE${NC}]"
}

#---------------------------------------------------------------------
# Function: ConfigureApacheForSOGo
#    Configure Apache for SOGo
#---------------------------------------------------------------------
ConfigureApacheForSOGo() {
    echo -e "$IDENTATION_LVL_2 Configuring Apache for SOGo..."

    # Create Apache configuration for SOGo
    cat > /etc/apache2/sites-available/sogo.conf <<EOF
<VirtualHost *:80>
    ServerName webmail.$CFG_HOSTNAME_DOMAIN
    DocumentRoot /usr/local/lib/GNUstep/SOGo/WebServerResources

    <Directory /usr/local/lib/GNUstep/SOGo/>
        AllowOverride None
        Require all granted
    </Directory>

    Alias /SOGo.woa/WebServerResources /usr/local/lib/GNUstep/SOGo/WebServerResources
    Alias /SOGo/WebServerResources /usr/local/lib/GNUstep/SOGo/WebServerResources

    <Location /SOGo>
        ProxyPass http://127.0.0.1:20000/SOGo retry=0
        ProxyPassReverse http://127.0.0.1:20000/SOGo
    </Location>

    ErrorLog \${APACHE_LOG_DIR}/sogo-error.log
    CustomLog \${APACHE_LOG_DIR}/sogo-access.log combined
</VirtualHost>

<IfModule mod_ssl.c>
<VirtualHost *:443>
    ServerName webmail.$CFG_HOSTNAME_DOMAIN
    DocumentRoot /usr/local/lib/GNUstep/SOGo/WebServerResources

    <Directory /usr/local/lib/GNUstep/SOGo/>
        AllowOverride None
        Require all granted
    </Directory>

    Alias /SOGo.woa/WebServerResources /usr/local/lib/GNUstep/SOGo/WebServerResources
    Alias /SOGo/WebServerResources /usr/local/lib/GNUstep/SOGo/WebServerResources

    <Location /SOGo>
        ProxyPass http://127.0.0.1:20000/SOGo retry=0
        ProxyPassReverse http://127.0.0.1:20000/SOGo
    </Location>

    SSLEngine on
    SSLCertificateFile /home/$SUDO_USER/.acme.sh/webmail.$CFG_HOSTNAME_DOMAIN/fullchain.cer
    SSLCertificateKeyFile /home/$SUDO_USER/.acme.sh/webmail.$CFG_HOSTNAME_DOMAIN/webmail.$CFG_HOSTNAME_DOMAIN.key
    SSLCipherSuite ECDHE-ECDSA-AES256-GCM-SHA384:ECDHE-RSA-AES256-GCM-SHA384:ECDHE-ECDSA-CHACHA20-POLY1305:ECDHE-RSA-CHACHA20-POLY1305:ECDHE-ECDSA-AES128-GCM-SHA256:ECDHE-RSA-AES128-GCM-SHA256:DHE-RSA-AES256-GCM-SHA384:DHE-RSA-AES128-GCM-SHA256
    SSLHonorCipherOrder on

    ErrorLog \${APACHE_LOG_DIR}/sogo-error.log
    CustomLog \${APACHE_LOG_DIR}/sogo-access.log combined
</VirtualHost>
</IfModule>
EOF

    # Enable required Apache modules
    a2enmod proxy proxy_http ssl
    a2ensite sogo.conf
    systemctl restart apache2

    echo -e "$IDENTATION_LVL_2 Apache configuration for SOGo completed. [${green}DONE${NC}]"
}

#---------------------------------------------------------------------
# Function: ConfigureNginxForSOGo
#    Configure Nginx for SOGo
#---------------------------------------------------------------------
ConfigureNginxForSOGo() {
    echo -e "$IDENTATION_LVL_2 Configuring Nginx for SOGo..."

    # Create Nginx configuration for SOGo
    cat > /etc/nginx/sites-available/sogo.conf <<EOF
server {
    listen 80;
    server_name webmail.$CFG_HOSTNAME_DOMAIN;
    return 301 https://\$server_name\$request_uri;
}

server {
    listen 443 ssl http2;
    server_name webmail.$CFG_HOSTNAME_DOMAIN;

    ssl_certificate /home/$SUDO_USER/.acme.sh/webmail.$CFG_HOSTNAME_DOMAIN/fullchain.cer;
    ssl_certificate_key /home/$SUDO_USER/.acme.sh/webmail.$CFG_HOSTNAME_DOMAIN/webmail.$CFG_HOSTNAME_DOMAIN.key;
    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_ciphers 'ECDHE-ECDSA-AES256-GCM-SHA384:ECDHE-RSA-AES256-GCM-SHA384:ECDHE-ECDSA-CHACHA20-POLY1305:ECDHE-RSA-CHACHA20-POLY1305:ECDHE-ECDSA-AES128-GCM-SHA256:ECDHE-RSA-AES128-GCM-SHA256:DHE-RSA-AES256-GCM-SHA384:DHE-RSA-AES128-GCM-SHA256';
    ssl_prefer_server_ciphers on;
    ssl_session_timeout 1d;
    ssl_session_cache shared:SSL:50m;
    ssl_session_tickets off;

    # HSTS configuration
    add_header Strict-Transport-Security "max-age=63072000" always;

    location /SOGo {
        proxy_pass http://127.0.0.1:20000/SOGo;
        proxy_set_header Host \$host;
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto \$scheme;
    }

    location /SOGo.woa/WebServerResources {
        alias /usr/local/lib/GNUstep/SOGo/WebServerResources;
    }

    location /SOGo/WebServerResources {
        alias /usr/local/lib/GNUstep/SOGo/WebServerResources;
    }
}
EOF

    # Enable the site and restart Nginx
    ln -s /etc/nginx/sites-available/sogo.conf /etc/nginx/sites-enabled/
    systemctl restart nginx

    echo -e "$IDENTATION_LVL_2 Nginx configuration for SOGo completed. [${green}DONE${NC}]"
}

#---------------------------------------------------------------------
# Function: InstallSquirrelmail
#    Install and configure Squirrelmail
#---------------------------------------------------------------------
InstallSquirrelmail() {
    echo -e "$IDENTATION_LVL_1 Installing ${red}SquirrelMail${NC} WebMail Client... "

    echo "dictionaries-common dictionaries-common/default-wordlist select american (American English)" | debconf-set-selections
    apt_install squirrelmail wamerican
	
    if [ $CFG_WEBSERVER == "apache" ]; then
	ln -s /etc/squirrelmail/apache.conf /etc/apache2/conf-available/squirrelmail.conf
	a2enconf squirrelmail
	sed -i 1d /etc/squirrelmail/apache.conf
	sed -i '1iAlias /webmail /usr/share/squirrelmail' /etc/squirrelmail/apache.conf

    elif [ $CFG_WEBSERVER == "nginx" ]; then        
        # Create nginx configuration for SquirrelMail
        cat > /etc/nginx/conf.d/squirrelmail.conf <<EOF
location /webmail {
    alias /usr/share/squirrelmail;
    index index.php;
    
    location ~ ^/webmail/(.+\.php)$ {
        alias /usr/share/squirrelmail/\$1;
        fastcgi_pass unix:/var/run/php/php-fpm.sock;
        include fastcgi_params;
        fastcgi_param SCRIPT_FILENAME \$request_filename;
    }

    location ~* ^/webmail/(.+\.(jpg|jpeg|gif|css|png|js|ico|html|xml|txt))$ {
        alias /usr/share/squirrelmail/\$1;
    }
}
EOF

        # Reload nginx to apply changes
        systemctl reload nginx
    fi

    # Configure based on MTA (same for both web servers)
    case $CFG_MTA in
        "courier")
            sed -i 's/$imap_server_type       = "other";/$imap_server_type       = "courier";/' /etc/squirrelmail/config.php
            sed -i 's/$optional_delimiter     = "detect";/$optional_delimiter     = ".";/' /etc/squirrelmail/config.php
            sed -i 's/$default_folder_prefix          = "";/$default_folder_prefix          = "INBOX.";/' /etc/squirrelmail/config.php
            sed -i 's/$trash_folder                   = "INBOX.Trash";/$trash_folder                   = "Trash";/' /etc/squirrelmail/config.php
            sed -i 's/$sent_folder                    = "INBOX.Sent";/$sent_folder                    = "Sent";/' /etc/squirrelmail/config.php
            sed -i 's/$draft_folder                   = "INBOX.Drafts";/$draft_folder                   = "Drafts";/' /etc/squirrelmail/config.php
            sed -i 's/$default_sub_of_inbox           = true;/$default_sub_of_inbox           = false;/' /etc/squirrelmail/config.php
            sed -i 's/$delete_folder                  = false;/$delete_folder                  = true;/' /etc/squirrelmail/config.php
            ;;
        "dovecot")
            sed -i 's/$imap_server_type       = "other";/$imap_server_type       = "dovecot";/' /etc/squirrelmail/config.php
            sed -i 's/$trash_folder                   = "INBOX.Trash";/$trash_folder                   = "Trash";/' /etc/squirrelmail/config.php
            sed -i 's/$sent_folder                    = "INBOX.Sent";/$sent_folder                    = "Sent";/' /etc/squirrelmail/config.php
            sed -i 's/$draft_folder                   = "INBOX.Drafts";/$draft_folder                   = "Drafts";/' /etc/squirrelmail/config.php
            sed -i 's/$default_sub_of_inbox           = true;/$default_sub_of_inbox           = false;/' /etc/squirrelmail/config.php
            sed -i 's/$delete_folder                  = false;/$delete_folder                  = true;/' /etc/squirrelmail/config.php
            ;;
    esac
    echo -e "[${green}DONE${NC}]"
}
