#---------------------------------------------------------------------
# Function: InstallOwncloud and Nextcloud
#    Install and configure of Owncloud and Nextcloud
#---------------------------------------------------------------------

InstallNextcloud() {

    version=$1
    hash=$2

    echo
    echo "Upgrading to Nextcloud version $version"
    echo

    # Remove the current owncloud/Nextcloud
    rm -rf $STORAGE_ROOT/owncloud

    # Download and verify
    wget_verify https://download.nextcloud.com/server/releases/nextcloud-$version.zip $hash /tmp/nextcloud.zip

    # Extract ownCloud/Nextcloud
    unzip -q /tmp/nextcloud.zip -d $STORAGE_ROOT
    mv $STORAGE_ROOT/nextcloud $STORAGE_ROOT/owncloud
    rm -f /tmp/nextcloud.zip

    # The two apps we actually want are not in Nextcloud core. Download the releases from
    # their github repositories.
    mkdir -p $STORAGE_ROOT/owncloud/apps

    # Admin
    # https://apps.nextcloud.com/apps/onlyoffice
    wget_verify https://github.com/ONLYOFFICE/onlyoffice-nextcloud/releases/download/v2.1.2/onlyoffice.tar.gz 96e520ec1e219f4359e3743370d251ce5122819e /tmp/onlyoffice.tgz
    tar xf /tmp/onlyoffice.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/onlyoffice.tgz

    # https://apps.nextcloud.com/apps/richdocuments
    wget_verify https://github.com/nextcloud/richdocuments/releases/download/v3.1.1/richdocuments.tar.gz 2e5c8ae76a7571ecb3aba410c4c4a440e47abef2 /tmp/richdocuments.tgz
    tar xf /tmp/richdocuments.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/richdocuments.tgz

    # https://apps.nextcloud.com/apps/jsloader
    wget_verify https://github.com/nextcloud/jsloader/releases/download/v1.0.2/jsloader-1.0.2.tar.gz 38273a83ad2f35bd75b84e5b9220ca685d25e6ca /tmp/jsloader.tgz
    tar xf /tmp/jsloader.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/jsloader.tgz

    # https://apps.nextcloud.com/apps/drop_account
    wget_verify https://framagit.org/tcit/drop_user/uploads/d10892f2ebaa5933ea96b9b47c28bc32/drop_account.tar.gz 14b547cdb4ceabe8d65461f7ac57deeed11c7efb /tmp/drop_account.tgz
    tar xf /tmp/drop_account.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/drop_account.tgz

    # https://apps.nextcloud.com/apps/files_accesscontrol
    wget_verify https://github.com/nextcloud/files_accesscontrol/releases/download/v1.5.0/files_accesscontrol-1.5.0.tar.gz 2905fde842a1fb8829a8594d37efae6c38d4db8c /tmp/files_accesscontrol.tgz
    tar xf /tmp/files_accesscontrol.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/files_accesscontrol.tgz

    # https://apps.nextcloud.com/apps/bruteforcesettings
    wget_verify https://github.com/nextcloud/bruteforcesettings/releases/download/v1.3.0/bruteforcesettings-1.3.0.tar.gz 3d10fa15ef31488cd72e0c9137f13e2acfde81eb /tmp/bruteforcesettings.tgz
    tar xf /tmp/bruteforcesettings.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/bruteforcesettings.tgz

    # https://apps.nextcloud.com/apps/files_antivirus
    wget_verify https://github.com/nextcloud/files_antivirus/archive/master.tar.gz adb01d3577bc06b9dd02c2c42ce0719cbecc1eba /tmp/files_antivirus.tgz
    tar xf /tmp/files_antivirus.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/files_antivirus.tgz

    # https://apps.nextcloud.com/apps/ransomware_protection
    wget_verify https://github.com/nextcloud/ransomware_protection/releases/download/v1.3.0/ransomware_protection-1.3.0.tar.gz d67888c25b11165bfb91a03381dc283050f13245 /tmp/ransomware_protection.tgz
    tar xf /tmp/ransomware_protection.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/ransomware_protection.tgz

    # Admin apps
    # https://apps.nextcloud.com/apps/sociallogin
    # 'social_login_auto_redirect' => true setting in config.php
    wget_verify https://github.com/zorn-v/nextcloud-social-login/releases/download/v1.12.0/release.tar.gz 9591913586d5a99caaacd26e9917b1d1f0fff39d /tmp/sociallogin.tgz
    tar xf /tmp/sociallogin.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/sociallogin.tgz

    # https://apps.nextcloud.com/apps/external
    wget_verify https://github.com/nextcloud/external/releases/download/v3.2.0/external-3.2.0.tar.gz d8b5ec2c9911ebf4b8a5a6cb53b1f54336c0a4ec /tmp/external.tgz
    tar xf /tmp/external.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/external.tgz

    # https://apps.nextcloud.com/apps/sharepoint
    wget_verify https://github.com/nextcloud/external/releases/download/v1.3.0/sharepoint-1.3.0.tar.gz 7ca83b57d056fea24e7677af811cfcbf59e030ba /tmp/sharepoint.tgz
    tar xf /tmp/sharepoint.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/sharepoint.tgz

    # https://apps.nextcloud.com/apps/announcementcenter
    wget_verify https://github.com/nextcloud/announcementcenter/releases/download/v3.4.0/announcementcenter-3.4.0.tar.gz d1197b7cc4a9276cf05acd6250236e0d654a936e /tmp/announcementcenter.tgz
    tar xf /tmp/announcementcenter.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/announcementcenter.tgz

    # https://apps.nextcloud.com/apps/impersonate
    wget_verify https://github.com/nextcloud/impersonate/releases/download/v1.2.0/impersonate-1.2.0.tar.gz d1394787a96d785c73916e4695ecc0f86ae8d7b9 /tmp/impersonate.tgz
    tar xf /tmp/impersonate.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/impersonate.tgz

    # https://apps.nextcloud.com/apps/ocdownloader
    wget_verify https://github.com/e-alfred/ocdownloader/releases/download/1.6.0-Alpha2/ocdownloader_1.6.0alpha2 59625f5ec936ab1c26744f222718124848944d9d /tmp/ocdownloader.tgz
    tar xf /tmp/ocdownloader.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/ocdownloader.tgz

    # https://apps.nextcloud.com/apps/files_retention
    wget_verify https://github.com/nextcloud/files_retention/releases/download/v1.4.0/files_retention-1.4.0.tar.gz 6ebca50203739ee95ab8de93d8f8f2095a8ffd7d /tmp/files_retention.tgz
    tar xf /tmp/files_retention.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/files_retention.tgz

    # https://apps.nextcloud.com/apps/files_automatedtagging
    wget_verify https://github.com/nextcloud/files_automatedtagging/releases/download/v1.5.0/files_automatedtagging-1.5.0.tar.gz 4d3325d530754a02a487dab20c67988d2f2a0d7d /tmp/files_automatedtagging.tgz
    tar xf /tmp/files_automatedtagging.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/files_automatedtagging.tgz

    # https://apps.nextcloud.com/apps/beame_insta_ssl
    #wget_verify https://github.com/nextcloud/beame_insta_ssl/releases/download/v0.0.6/beame_insta_ssl.tar.gz 48eb743ff3a2d3e29c19975873625b033eb6bd7a /tmp/beame_insta_ssl.tgz
    #tar xf /tmp/beame_insta_ssl.tgz -C $STORAGE_ROOT/owncloud/apps/
    #rm /tmp/beame_insta_ssl.tgz

    # https://apps.nextcloud.com/apps/groupfolders
    wget_verify https://github.com/nextcloud/groupfolders/releases/download/v2.0.2/groupfolders-2.0.2.tar.gz 67bcc00711f6d11f2c713916ea283cdaa21a366e /tmp/groupfolders.tgz
    tar xf /tmp/groupfolders.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/groupfolders.tgz

    # https://apps.nextcloud.com/apps/checksum
    wget_verify https://github.com/westberliner/owncloud-checksum/releases/download/v0.4.2/checksum.tar.gz c3b8e1c52296750c6f4b0f4c18f6316eb8134c17 /tmp/checksum.tgz
    tar xf /tmp/checksum.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/checksum.tgz

    # https://apps.nextcloud.com/apps/files_accesscontrol
    wget_verify https://github.com/nextcloud/files_accesscontrol/releases/download/v1.5.0/files_accesscontrol-1.5.0.tar.gz 2905fde842a1fb8829a8594d37efae6c38d4db8c /tmp/files_accesscontrol.tgz
    tar xf /tmp/files_accesscontrol.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/files_accesscontrol.tgz

    # https://apps.nextcloud.com/apps/nextant
    #wget_verify https://github.com/nextcloud/nextant/releases/download/v1.0.8/nextant-1.0.8.tar.gz ebfbcb028583608e3fa7b9697facc626253dd002 /tmp/nextant.tgz
    #tar xf /tmp/nextant.tgz -C $STORAGE_ROOT/owncloud/apps/
    #rm /tmp/nextant.tgz

    # https://apps.nextcloud.com/apps/fulltextsearch
    wget_verify https://github.com/nextcloud/fulltextsearch/releases/download/v1.2.3/fulltextsearch-1.2.3.tar.gz 68b554dc185d99f6c7679d789260f75f7b8f0dee /tmp/fulltextsearch.tgz
    tar xf /tmp/fulltextsearch.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/fulltextsearch.tgz

    # https://apps.nextcloud.com/apps/defaultgroup
    #wget_verify https://github.com/Stibila/defaultgroup/releases/download/v0.3.0/defaultgroup.tar.gz e6e379646c344b6115efc84db2f3bef0d62582d9 /tmp/defaultgroup.tgz
    #tar xf /tmp/defaultgroup.tgz -C $STORAGE_ROOT/owncloud/apps/
    #rm /tmp/defaultgroup.tgz

    #wget_verify https://github.com/Patt92/Workin2Gether/releases/download/0.9.6/workin2gether.tar.gz ff39d5c924abdf19a989ed328c6ecaabcf58d152 /tmp/workin2gether.tgz
    #tar xf /tmp/workin2gether.tgz -C $STORAGE_ROOT/owncloud/apps/
    #rm /tmp/workin2gether.tgz

    # https://apps.nextcloud.com/apps/quota_warning
    wget_verify https://github.com/nextcloud/quota_warning/releases/download/v1.4.0/quota_warning-1.4.0.tar.gz 39bfc9f2d411e3d31fb65dd9a807d5e1ee2bb2ee /tmp/quota_warning.tgz
    tar xf /tmp/quota_warning.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/quota_warning.tgz

    # https://apps.nextcloud.com/apps/admin_notifications
    #wget_verify https://github.com/nextcloud/admin_notifications/releases/download/v1.0.0/admin_notifications-1.0.0.tar.gz 534c96dfb5f4aa24e4a2239e08ec6949050811c9 /tmp/admin_notifications.tgz
    #tar xf /tmp/admin_notifications.tgz -C $STORAGE_ROOT/owncloud/apps/
    #rm /tmp/admin_notifications.tgz

    # https://apps.nextcloud.com/apps/user_usage_report
    wget_verify https://github.com/nextcloud/user_usage_report/releases/download/v1.1.1/user_usage_report-1.1.1.tar.gz c0df530372e17880fb86521ec64a54088ea6132f /tmp/user_usage_report.tgz
    tar xf /tmp/user_usage_report.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/user_usage_report.tgz

    # https://apps.nextcloud.com/apps/apporder
    wget_verify https://github.com/juliushaertl/apporder/releases/download/v0.6.0/apporder.tar.gz 2fe440ea3a3dba19fb6645f6690b5645fefbfa22 /tmp/apporder.tgz
    tar xf /tmp/apporder.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/apporder.tgz

    # https://apps.nextcloud.com/apps/unsplash
    wget_verify https://github.com/jancborchardt/unsplash/releases/download/v1.1.3/unsplash-1.1.3.tar.gz e4f8c62067cbf657fbff80f744b9991e55448520 /tmp/unsplash.tgz
    tar xf /tmp/unsplash.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/unsplash.tgz

    # https://apps.nextcloud.com/apps/twofactor_u2f
    wget_verify https://github.com/nextcloud/twofactor_u2f/releases/download/v2.1.0/twofactor_u2f.tar.gz e1a1c2f7563934d06dafb135e8b2893cf3e29150 /tmp/twofactor_u2f.tgz
    tar xf /tmp/twofactor_u2f.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/twofactor_u2f.tgz

    # https://apps.nextcloud.com/apps/twofactor_totp
    wget_verify https://github.com/nextcloud/twofactor_totp/releases/download/v2.1.0/twofactor_totp.tar.gz 57e12de1b587a49dff2e5a8479408cad41adfdfc  /tmp/twofactor_totp.tgz
    tar xf /tmp/twofactor_totp.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/twofactor_totp.tgz

    # https://apps.nextcloud.com/apps/twofactor_admin
    wget_verify https://github.com/ChristophWurst/twofactor_admin/releases/download/v0.2.0/twofactor_admin.tar.gz c669e1c0c0323702d6a44d404f00a543cbb5d941  /tmp/twofactor_admin.tgz
    tar xf /tmp/twofactor_admin.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/twofactor_admin.tgz

    # https://apps.nextcloud.com/apps/twofactor_nextcloud_notification
    wget_verify https://github.com/nickv-nextcloud/twofactor_nextcloud_notification/releases/download/v1.0.2/twofactor_nextcloud_notification.tar.gz bfc471656293941cd477925b98c4c7baafe35263  /tmp/twofactor_nextcloud_notification.tgz
    tar xf /tmp/twofactor_nextcloud_notification.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/twofactor_nextcloud_notification.tgz

    # basic apps
    # https://apps.nextcloud.com/apps/dashboard
    wget_verify https://github.com/nextcloud/dashboard/releases/download/v6.0.0/dashboard-6.0.0.tar.gz 8666a1106cc60d0ffcf65e9f204084b8d06340a0 /tmp/dashboard.tgz
    tar xf /tmp/dashboard.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/dashboard.tgz

    # https://apps.nextcloud.com/apps/contacts
    wget_verify https://github.com/nextcloud/contacts/releases/download/v3.0.1/contacts.tar.gz fdc7b9f419e197f49bc311296dbd71ad9d2db25e /tmp/contacts.tgz
    tar xf /tmp/contacts.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/contacts.tgz

    # https://apps.nextcloud.com/apps/calendar
    wget_verify https://github.com/nextcloud/calendar/releases/download/v1.6.4/calendar.tar.gz d8a7950dba14803472b6c19625a8ceb23d6fd4ef /tmp/calendar.tgz
    tar xf /tmp/calendar.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/calendar.tgz

    # https://apps.nextcloud.com/apps/event_update_notification
    wget_verify https://github.com/nickv-nextcloud/event_update_notification/releases/download/v0.3.2/event_update_notification-0.3.2.tar.gz 70f090d3670a6763e902f06b04945f00a0bee0cd /tmp/event_update_notification.tgz
    tar xf /tmp/event_update_notification.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/event_update_notification.tgz

    # https://apps.nextcloud.com/apps/notes
    wget_verify https://github.com/nextcloud/notes/releases/download/v2.5.1/notes.tar.gz d96716ab37a86df0b76cc54e8ea0134cab0e6fa4 /tmp/notes.tgz
    tar xf /tmp/notes.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/notes.tgz

    # https://apps.nextcloud.com/apps/notes
    wget_verify https://github.com/nextcloud/notes/releases/download/v2.5.1/notes.tar.gz d96716ab37a86df0b76cc54e8ea0134cab0e6fa4 /tmp/notes.tgz
    tar xf /tmp/notes.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/notes.tgz

    # https://apps.nextcloud.com/apps/tasks
    wget_verify https://github.com/nextcloud/tasks/releases/download/v0.9.8/tasks.tar.gz 11b1320a72f0b31ace27f13474d3639115817bec /tmp/tasks.tgz
    tar xf /tmp/tasks.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/tasks.tgz

    # https://apps.nextcloud.com/apps/deck
    wget_verify https://github.com/nextcloud/deck/releases/download/v0.5.2/deck.tar.gz 7e7c6e374014e31b972e15b4ddf5674fc3cb5b9a /tmp/deck.tgz
    tar xf /tmp/deck.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/deck.tgz

    # GPS apps
    # https://apps.nextcloud.com/apps/phonetrack
    wget_verify https://gitlab.com/eneiluj/phonetrack-oc/wikis/uploads/aa2bc2061b8451285f3c3d826af1c74f/phonetrack-0.4.0.tar.gz 30e59d3c4036b384a5953b97645fe82be8848a56 /tmp/phonetrack.tgz
    tar xf /tmp/phonetrack.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/phonetrack.tgz

    # https://apps.nextcloud.com/apps/gpxpod
    wget_verify https://gitlab.com/eneiluj/gpxpod-oc/wikis/uploads/3f5fbc895de15f11f90582c4872839dc/gpxpod-3.0.1.tar.gz 0d9ffc1b915a632ff79e2ea4497780ae9bc9d3d7 /tmp/gpxpod.tgz
    tar xf /tmp/gpxpod.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/gpxpod.tgz

    # https://apps.nextcloud.com/apps/gpxmotion
    wget_verify https://gitlab.com/eneiluj/gpxmotion-oc/wikis/uploads/c057b6042c3185a937fa1a91fa1d1c96/gpxmotion-0.0.8.tar.gz 8180046a5b23a6e5ca74d041d74558933a09f5d4 /tmp/gpxmotion.tgz
    tar xf /tmp/gpxmotion.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/gpxmotion.tgz

    # https://apps.nextcloud.com/apps/gpxedit
    wget_verify https://gitlab.com/eneiluj/gpxedit-oc/wikis/uploads/33d187268c5f6f6a55350d656305701c/gpxedit-0.0.10.tar.gz 0b1cb2ffb0fedbdedcbf45c34c8f6733815aac7c /tmp/gpxedit.tgz
    tar xf /tmp/gpxedit.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/gpxedit.tgz

    # Password apps
    # https://apps.nextcloud.com/apps/passman
    wget_verify https://releases.passman.cc/passman_2.2.1.tar.gz 3d38d5e13c5475295027c01334686b0508b79bbc /tmp/passman.tgz
    tar xf /tmp/passman.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/passman.tgz

    # https://apps.nextcloud.com/apps/keeweb
    #wget_verify https://github.com/jhass/nextcloud-keeweb/releases/download/v0.4.0/keeweb-0.4.0.tar.gz b99c5500bcb0bf224f2ae31ff7393a88fbfdc40b /tmp/keeweb.tgz
    #tar xf /tmp/keeweb.tgz -C $STORAGE_ROOT/owncloud/apps/
    #rm /tmp/keeweb.tgz

    # https://apps.nextcloud.com/apps/passwords
    wget_verify https://git.mdns.eu/nextcloud/passwords/-/jobs/7374/artifacts/raw/passwords.tar.gz 82a30b73f67fbb9f9dd8edf627179782e58d87be /tmp/passwords.tgz
    tar xf /tmp/keeweb.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/passwords.tgz

    # Social sharing apps
    # https://apps.nextcloud.com/apps/socialsharing_email
    wget_verify https://github.com/nextcloud/socialsharing/releases/download/email-1.0.5/socialsharing_email.tar.gz 6aa00e5593437dca06098f1f4a1216369e0ac255 /tmp/socialsharing_email.tgz
    tar xf /tmp/socialsharing_email.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/socialsharing_email.tgz

    # https://apps.nextcloud.com/apps/socialsharing_facebook
    wget_verify https://github.com/nextcloud/socialsharing/releases/download/facebook-1.0.4/socialsharing_facebook.tar.gz c676fa502e86cd73a8fefd40880ff3a843182106 /tmp/socialsharing_facebook.tgz
    tar xf /tmp/socialsharing_facebook.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/socialsharing_facebook.tgz

    # https://apps.nextcloud.com/apps/socialsharing_twitter
    wget_verify https://github.com/nextcloud/socialsharing/releases/download/twitter-1.0.4/socialsharing_twitter.tar.gz eff8ebe2d31be6f6f9cd33d9e558ca1dbd3a3af8 /tmp/socialsharing_twitter.tgz
    tar xf /tmp/socialsharing_twitter.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/socialsharing_twitter.tgz

    # https://apps.nextcloud.com/apps/socialsharing_googleplus
    wget_verify https://github.com/nextcloud/socialsharing/releases/download/googleplus-1.0.4/socialsharing_googleplus.tar.gz 1bc648c8485f06dc223c5af1541832ee95804c97 /tmp/socialsharing_googleplus.tgz
    tar xf /tmp/socialsharing_googleplus.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/socialsharing_googleplus.tgz

    # https://apps.nextcloud.com/apps/socialsharing_diaspora
    wget_verify https://github.com/nextcloud/socialsharing/releases/download/diaspora-1.0.4/socialsharing_diaspora.tar.gz d35aa067348f5d9129de1494934e576fd58d49bd /tmp/socialsharing_googleplus.tgz
    tar xf /tmp/socialsharing_googleplus.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/socialsharing_googleplus.tgz

    # Bookmarks, Phone, Mail apps
    # https://apps.nextcloud.com/apps/bookmarks
    wget_spec_verify https://github.com/nextcloud/bookmarks/releases/download/v0.15.1/bookmarks-0.15.1.tar.gz 5731aaf609e333131b9c0545f16d6aff76815195 /tmp/bookmarks.tgz
    tar xf /tmp/bookmarks.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/bookmarks.tgz

    # https://apps.nextcloud.com/apps/ocsms
    wget_verify https://github.com/nextcloud/ocsms/releases/download/2.1.1/ocsms-2.1.1.tar.gz c8039e40b70b4d31e031aef82b8ee0c16690efc9 /tmp/ocsms.tgz
    tar xf /tmp/ocsms.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/ocsms.tgz

    # https://apps.nextcloud.com/apps/telephoneprovider
    wget_verify https://github.com/georgehrke/telephoneprovider/releases/download/v1.0.2/telephoneprovider.tar.gz 1b4a99c90b5c2bb7094d72e533cac9dc70cb9701 /tmp/telephoneprovider.tgz
    tar xf /tmp/telephoneprovider.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/telephoneprovider.tgz

    # https://apps.nextcloud.com/apps/rainloop
    wget_verify https://github.com/pierre-alain-b/rainloop-nextcloud/releases/download/6.0.2/rainloop.tar.gz 621d7f05ad77387f528994ca0a323ffb406d623f /tmp/rainloop.tgz
    tar xf /tmp/rainloop.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/rainloop.tgz

    # https://apps.nextcloud.com/apps/mail
    wget_verify https://github.com/nextcloud/mail/releases/download/v0.11.0/mail.tar.gz 0b84398a8cd538bcd4061a8ed6cdee791b9f7888 /tmp/mail.tgz
    tar xf /tmp/mail.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/mail.tgz

    # Video call
    # https://apps.nextcloud.com/apps/spreedme
    #wget_verify https://github.com/nextcloud/spreed/releases/download/v2.0.1/spreed-2.0.1.tar.gz 628ca2c8e7e207e236c37b52c77af3678d31cdfc  /tmp/spreed.tgz
    #tar xf /tmp/spreed.tgz -C $STORAGE_ROOT/owncloud/apps/
    #rm /tmp/spreed.tgz

    # https://apps.nextcloud.com/apps/spreed
    wget_verify https://github.com/nextcloud/spreed/releases/download/v5.0.0/spreed-5.0.0.tar.gz a07ed4590bb7038c9b39df987a55a85236290023  /tmp/spreed.tgz
    tar xf /tmp/spreed.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/spreed.tgz

    # Fix weird permissions.
    chmod 750 $STORAGE_ROOT/owncloud/{apps,config}

    # Create a symlink to the config.php in STORAGE_ROOT (for upgrades we're restoring the symlink we previously
    # put in, and in new installs we're creating a symlink and will create the actual config later).
    #ln -sf $STORAGE_ROOT/owncloud/config.php $STORAGE_ROOT/owncloud/config/config.php

    # Make sure permissions are correct or the upgrade step won't run.
    # $STORAGE_ROOT/owncloud may not yet exist, so use -f to suppress
    # that error.
    chown -f -R www-data.www-data $STORAGE_ROOT/owncloud $STORAGE_ROOT/owncloud

    # If this isn't a new installation, immediately run the upgrade script.
    # Then check for success (0=ok and 3=no upgrade needed, both are success).
    if [ -e $STORAGE_ROOT/owncloud/owncloud.db ]; then
	# ownCloud 8.1.1 broke upgrades. It may fail on the first attempt, but
	# that can be OK.
	sudo -u www-data php $STORAGE_ROOT/owncloud/occ upgrade
	if [ \( $? -ne 0 \) -a \( $? -ne 3 \) ]; then
	    echo "Trying ownCloud upgrade again to work around ownCloud upgrade bug..."
	    sudo -u www-data php $STORAGE_ROOT/owncloud/occ upgrade
	    if [ \( $? -ne 0 \) -a \( $? -ne 3 \) ]; then exit 1; fi
	    sudo -u www-data php $STORAGE_ROOT/owncloud/occ maintenance:mode --off
	    echo "...which seemed to work."
	fi
    fi
	
    #if [ $CFG_WEBSERVER == "apache" ]; then
    case $CFG_WEBSERVER in
    "Apache")
	#mv /etc/nextcloud/apache.conf /etc/nextcloud/apache.conf.default
	cat >> /etc/apache2/sites-available/nextcloud.conf <<EOF
<VirtualHost *:80>
 DocumentRoot "$STORAGE_ROOT/owncloud"
 ServerName cloud.wolftronics.hu

 ErrorLog ${APACHE_LOG_DIR}/error.log
 CustomLog ${APACHE_LOG_DIR}/access.log combined

<Directory $STORAGE_ROOT/owncloud/>
 Options +FollowSymlinks
 AllowOverride All

 <IfModule mod_dav.c>
 Dav off
 </IfModule>

 SetEnv HOME $STORAGE_ROOT/owncloud/>
 SetEnv HTTP_HOME $STORAGE_ROOT/owncloud
 Satisfy Any

</Directory>

</VirtualHost>
EOF

	cat >> /etc/apache2/sites-available/nextcloud-le-ssl.conf <<EOF
<IfModule mod_headers.c>
     Header always set Strict-Transport-Security "max-age=15768000; preload"
 </IfModule>
EOF
	
	ln -s /etc/apache2/sites-available/nextcloud.conf /etc/apache2/sites-enabled/nextcloud.conf
	a2enmod rewrite headers env dir mime setenvif ssl
	a2ensite default-ssl
	restart_service apache2

	;;
    "Nginx")

	#mkdir -p /etc/nginx/sites-available
	cat > /etc/nginx/sites-available/nextcloud.vhost <<EOF;
server {
    listen 80;
    server_name cloud.wolftronics.hu;
    return 301 https://\$server_name\$request_uri;
}

server {
#    listen 80;
    listen 443 ssl http2;
    server_name cloud.wolftronics.hu;

    #ssl_certificate     /etc/letsencrypt/live/cloud.wolftronics.hu/fullchain.pem;
    #ssl_certificate_key /etc/letsencrypt/live/cloud.wolftronics.hu/privkey.pem;
    ssl_certificate     /root/.acme.sh/cloud.wolftronics.hu/cloud.wolftronics.hu.cer;
    ssl_certificate_key /root/.acme.sh/cloud.wolftronics.hu/cloud.wolftronics.hu.key;

    ssl_session_timeout 1d;
    ssl_session_cache shared:SSL:10m;
    ssl_session_tickets off;

    # Add headers to serve security related headers
    # Before enabling Strict-Transport-Security headers please read into this
    # topic first.
    add_header Strict-Transport-Security "max-age=15768000;preload" always;
    add_header X-Content-Type-Options nosniff;
    add_header X-Frame-Options "SAMEORIGIN";
    add_header X-XSS-Protection "1; mode=block";
    add_header X-Robots-Tag none;
    add_header X-Download-Options noopen;
    add_header X-Permitted-Cross-Domain-Policies none;

    # Path to the root of your installation
    root $STORAGE_ROOT/owncloud/;

    location = /robots.txt {
        allow all;
        log_not_found off;
        access_log off;
    }

    # The following 2 rules are only needed for the user_webfinger app.
    # Uncomment it if you're planning to use this app.
    #rewrite ^/.well-known/host-meta /public.php?service=host-meta last;
    #rewrite ^/.well-known/host-meta.json /public.php?service=host-meta-json
    # last;

    location = /.well-known/carddav {
        return 301 \$scheme://\$host/remote.php/dav;
    }
    location = /.well-known/caldav {
       return 301 \$scheme://\$host/remote.php/dav;
    }

    location ~ /.well-known/acme-challenge {
      allow all;
    }

    # set max upload size
    client_max_body_size 512M;
    fastcgi_buffers 64 4K;

    # Disable gzip to avoid the removal of the ETag header
    gzip off;

    # Uncomment if your server is build with the ngx_pagespeed module
    # This module is currently not supported.
    #pagespeed off;

    error_page 403 /core/templates/403.php;
    error_page 404 /core/templates/404.php;

    location / {
       rewrite ^ /index.php\$uri;
    }

    location ~ ^/(?:build|tests|config|lib|3rdparty|templates|data)/ {
       deny all;
    }
    location ~ ^/(?:\.|autotest|occ|issue|indie|db_|console) {
       deny all;
     }

    location ~ ^/(?:index|remote|public|cron|core/ajax/update|status|ocs/v[12]|updater/.+|ocs-provider/.+|core/templates/40[34])\.php(?:$|/) {
       include fastcgi_params;
       fastcgi_split_path_info ^(.+\.php)(/.*)$;
       fastcgi_param SCRIPT_FILENAME \$document_root\$fastcgi_script_name;
       fastcgi_param PATH_INFO \$fastcgi_path_info;
       #Avoid sending the security headers twice
       fastcgi_param modHeadersAvailable true;
       fastcgi_param front_controller_active true;
       fastcgi_pass unix:/run/php/php7.0-fpm.sock;
       #fastcgi_pass unix:/run/php/php7.1-fpm.sock;
       fastcgi_intercept_errors on;
       fastcgi_request_buffering off;
    }

    location ~ ^/(?:updater|ocs-provider)(?:$|/) {
       try_files \$uri/ =404;
       index index.php;
    }

    # Adding the cache control header for js and css files
    # Make sure it is BELOW the PHP block
    location ~* \.(?:css|js)$ {
        try_files \$uri /index.php\$uri\$is_args\$args;
        add_header Cache-Control "public, max-age=7200";
        # Add headers to serve security related headers (It is intended to
        # have those duplicated to the ones above)
        add_header X-Content-Type-Options nosniff;
        add_header X-Frame-Options "SAMEORIGIN";
        add_header X-XSS-Protection "1; mode=block";
        add_header X-Robots-Tag none;
        add_header X-Download-Options noopen;
        add_header X-Permitted-Cross-Domain-Policies none;
        # Optional: Don't log access to assets
        access_log off;
   }

   location ~* \.(?:svg|gif|png|html|ttf|woff|ico|jpg|jpeg)$ {
        try_files \$uri /index.php\$uri\$is_args\$args;
        # Optional: Don't log access to other assets
        access_log off;
   }
}
EOF

	mkdir -p /etc/nginx/sites-enabled
	ln -s /etc/nginx/sites-available/nextcloud.vhost /etc/nginx/sites-enabled/nextcloud.vhost
	;;
    esac

mkdir /var/www/owncloud-data/
chown www-data:www-data /var/www/owncloud-data -R

}

# We only install ownCloud intermediate versions to be able to seemlesly upgrade to Nextcloud
InstallOwncloud() {

    version=$1
    hash=$2

    echo
    echo "Upgrading to OwnCloud version $version"
    echo

    # Remove the current owncloud/Nextcloud
    rm -rf $STORAGE_ROOT/owncloud

    # Download and verify
    wget_verify https://download.owncloud.org/community/owncloud-$version.zip $hash /tmp/owncloud.zip


    # Extract ownCloud
    unzip -q /tmp/owncloud.zip -d $STORAGE_ROOT
    rm -f /tmp/owncloud.zip

    # The two apps we actually want are not in Nextcloud core. Download the releases from
    # their github repositories.
    mkdir -p $STORAGE_ROOT/owncloud/apps

    wget_verify https://github.com/owncloud/contacts/releases/download/v1.4.0.0/contacts.tar.gz c1c22d29699456a45db447281682e8bc3f10e3e7 /tmp/contacts.tgz
    tar xf /tmp/contacts.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/contacts.tgz

    wget_verify https://github.com/nextcloud/calendar/releases/download/v1.4.0/calendar.tar.gz c84f3170efca2a99ea6254de34b0af3cb0b3a821 /tmp/calendar.tgz
    tar xf /tmp/calendar.tgz -C $STORAGE_ROOT/owncloud/apps/
    rm /tmp/calendar.tgz

    # Fix weird permissions.
    chmod 750 $STORAGE_ROOT/owncloud/{apps,config}

    # Create a symlink to the config.php in STORAGE_ROOT (for upgrades we're restoring the symlink we previously
    # put in, and in new installs we're creating a symlink and will create the actual config later).
    #ln -sf $STORAGE_ROOT/owncloud/config.php $STORAGE_ROOT/owncloud/config/config.php

    # Make sure permissions are correct or the upgrade step won't run.
    # $STORAGE_ROOT/owncloud may not yet exist, so use -f to suppress
    # that error.
    #chown -f -R www-data.www-data $STORAGE_ROOT/owncloud $STORAGE_ROOT/owncloud

    # If this isn't a new installation, immediately run the upgrade script.
    # Then check for success (0=ok and 3=no upgrade needed, both are success).
    if [ -e $STORAGE_ROOT/owncloud/owncloud.db ]; then
	# ownCloud 8.1.1 broke upgrades. It may fail on the first attempt, but
	# that can be OK.
	sudo -u www-data php5 $STORAGE_ROOT/owncloud/occ upgrade
	if [ \( $? -ne 0 \) -a \( $? -ne 3 \) ]; then
	    echo "Trying ownCloud upgrade again to work around ownCloud upgrade bug..."
	    sudo -u www-data php5 $STORAGE_ROOT/owncloud/occ upgrade
	    if [ \( $? -ne 0 \) -a \( $? -ne 3 \) ]; then exit 1; fi
	    sudo -u www-data php5 $STORAGE_ROOT/owncloud/occ maintenance:mode --off
	    echo "...which seemed to work."
	fi
    fi
}

InstallCloud() {
STORAGE_ROOT="/var/www"
	
	echo -n "Installing Nextcloud (contacts/calendar)..."
	if [ "$CFG_CLOUDUPGRADE" == "yes" ]; then
	    # Keep the php5 dependancies for the owncloud upgrades
	    apt_install \
		dbconfig-common \
		php5-cli php5-sqlite php5-gd php5-imap php5-curl php-pear php-apc curl libapr1 libtool libcurl4-openssl-dev php-xml-parser \
		php5 php5-dev php5-gd php5-fpm memcached php5-memcached
	
	    apt-get purge -qq -y owncloud*
	fi
	    apt_install php7.0 php7.0-fpm \
		php7.0-cli php7.0-sqlite php7.0-gd php7.0-imap php7.0-curl php-pear curl \
		php7.0-dev memcached php-memcached php7.0-xml php7.0-mbstring php7.0-zip php-apcu \
		php7.0-fpm php7.0-mysql php7.0-mcrypt php7.0-xsl php7.0-bcmath php7.0-cgi php-gettext libxml2-dev \
		php7.0-json php7.0-mysql php7.0-intl php-imagick php7.0-common php7.0-opcache php7.0-readline php7.0
		# php-apc
	
	# Migrate <= v0.10 setups that stored the ownCloud config.php in /usr/local rather than
	# in STORAGE_ROOT. Move the file to STORAGE_ROOT.
	if [ ! -f $STORAGE_ROOT/owncloud/config.php ] \
	    && [ -f $STORAGE_ROOT/owncloud/config/config.php ]; then

	    # Move config.php and symlink back into previous location.
	    echo "Migrating owncloud/config.php to new location."
	    mv $STORAGE_ROOT/owncloud/config/config.php $STORAGE_ROOT/owncloud/config.php \
	    && \
	    ln -sf $STORAGE_ROOT/owncloud/config.php $STORAGE_ROOT/owncloud/config/config.php
	fi

owncloud_ver=15.0.2
owncloud_hash=a54ad9f1b143a5e7c89bf0fae3ba398900266d58

# Check if Nextcloud dir exist, and check if version matches owncloud_ver (if either doesn't - install/upgrade)
if [ ! -d $STORAGE_ROOT/owncloud/ ] \
        || ! grep -q $owncloud_ver $STORAGE_ROOT/owncloud/version.php; then

    # Stop php-fpm if running. If theyre not running (which happens on a previously failed install), dont bail.
    service php7.0-fpm stop &> /dev/null || /bin/true
    #service php7.1-fpm stop &> /dev/null || /bin/true
    service php5-fpm stop &> /dev/null || /bin/true

    # Backup the existing ownCloud/Nextcloud.
    # Create a backup directory to store the current installation and database to
    BACKUP_DIRECTORY=$STORAGE_ROOT/owncloud-backup/`date +"%Y-%m-%d-%T"`
    mkdir -p "$BACKUP_DIRECTORY"
    if [ -d $STORAGE_ROOT/owncloud/ ]; then
	echo "upgrading ownCloud/Nextcloud to $owncloud_flavor $owncloud_ver (backing up existing installation, configuration and database to directory to $BACKUP_DIRECTORY..."
	cp -r $STORAGE_ROOT/owncloud "$BACKUP_DIRECTORY/owncloud-install"
    fi
    if [ -e /home/user-data/owncloud/owncloud.db ]; then
	cp /home/user-data/owncloud/owncloud.db $BACKUP_DIRECTORY
    fi
    if [ -e /home/user-data/owncloud/config.php ]; then
	cp /home/user-data/owncloud/config.php $BACKUP_DIRECTORY
    fi

    # We only need to check if we do upgrades when owncloud/Nextcloud was previously installed
    if [ -e $STORAGE_ROOT/owncloud/version.php ]; then
	if grep -q "OC_VersionString = '8\.1\.[0-9]" $STORAGE_ROOT/owncloud/version.php; then
	    echo "We are running 8.1.x, upgrading to 8.2.3 first"
	    InstallOwncloud 8.2.3 bfdf6166fbf6fc5438dc358600e7239d1c970613
	fi

	# If we are upgrading from 8.2.x we should go to 9.0 first. Owncloud doesn't support skipping minor versions
	if grep -q "OC_VersionString = '8\.2\.[0-9]" $STORAGE_ROOT/owncloud/version.php; then
	    echo "We are running version 8.2.x, upgrading to 9.0.2 first"

	    # We need to disable memcached. The upgrade and install fails
	    # with memcached
	    CONFIG_TEMP=$(/bin/mktemp)
	    php << "EOF" > $CONFIG_TEMP && mv $CONFIG_TEMP $STORAGE_ROOT/owncloud/config.php;
	    <?php
		include("$STORAGE_ROOT/owncloud/config.php");
		$CONFIG['memcache.local'] = '\OC\Memcache\APCu';
		echo "<?php\n\\\$CONFIG = ";
		var_export($CONFIG);
		echo ";";
	    ?>
EOF
	    #chown www-data.www-data $STORAGE_ROOT/owncloud/config.php

	    # We can now install owncloud 9.0.2
	    InstallOwncloud 9.0.2 72a3d15d09f58c06fa8bee48b9e60c9cd356f9c5

	    # The owncloud 9 migration doesn't migrate calendars and contacts
	    # The option to migrate these are removed in 9.1
	    # So the migrations should be done when we have 9.0 installed
	    sudo -u www-data php5 $STORAGE_ROOT/owncloud/occ dav:migrate-addressbooks
	    # The following migration has to be done for each owncloud user
	    for directory in $STORAGE_ROOT/owncloud/*@*/ ; do
		username=$(basename "${directory}")
		sudo -u www-data php5 $STORAGE_ROOT/owncloud/occ dav:migrate-calendar $username
	    done
	    sudo -u www-data php5 $STORAGE_ROOT/owncloud/occ dav:sync-birthday-calendar
	fi

	# If we are upgrading from 9.0.x we should go to 9.1 first.
	if grep -q "OC_VersionString = '9\.0\.[0-9]" $STORAGE_ROOT/owncloud/version.php; then
	    echo "We are running ownCloud 9.0.x, upgrading to ownCloud 9.1.4 first"
	    InstallOwncloud 9.1.4 e637cab7b2ca3346164f3506b1a0eb812b4e841a
	fi

	# If we are upgrading from 9.1.x we should go to Nextcloud 10.0 first.
	if grep -q "OC_VersionString = '9\.1\.[0-9]" $STORAGE_ROOT/owncloud/version.php; then
	    echo "We are running ownCloud 9.1.x, upgrading to Nextcloud 10.0.6 first"
	    InstallNextcloud 10.0.6 bb911f417d91b9d0741f8d1b20c863a7d8de5419
	fi

	# If we are upgrading from 10.0.x we should go to Nextcloud 11.0 first.
	if grep -q "OC_VersionString = '10\.0\.[0-9]" $STORAGE_ROOT/owncloud/version.php; then
	echo "We are running Nextcloud 10.0.x, upgrading to Nextcloud 11.0.8 first"
	    InstallNextcloud 11.0.8 a6e001494216a787de306eacece61dbe15e411b9
	fi

	# If we are upgrading from 11.0.x we should go to Nextcloud 12.0 first.
	if grep -q "OC_VersionString = '11\.0\.[0-9]" $STORAGE_ROOT/owncloud/version.php; then
	echo "We are running Nextcloud 11.0.x, upgrading to Nextcloud 12.0.13 first"
	    InstallNextcloud 12.0.13 b3b268a5a96d5f5c288fbe7e935ef68011e17e45
	fi

	# If we are upgrading from 12.0.x we should go to Nextcloud 13.0 first.
	if grep -q "OC_VersionString = '12\.0\.[0-9]" $STORAGE_ROOT/owncloud/version.php; then
	echo "We are running Nextcloud 12.0.x, upgrading to Nextcloud 13.0.10 first"
	    InstallNextcloud 13.0.10 85390a3a17de7520512456e8cbd067a0a9f76c4e
	fi

	# If we are upgrading from 13.0.x we should go to Nextcloud 14.0 first.
	if grep -q "OC_VersionString = '13\.0\.[0-9]" $STORAGE_ROOT/owncloud/version.php; then
	echo "We are running Nextcloud 13.0.x, upgrading to Nextcloud 14.0.6 first"
	    InstallNextcloud 14.0.6 4e43a57340f04c2da306c8eea98e30040399ae5a
	fi

	# If we are upgrading from 14.0.x we should go to Nextcloud 15.0 first.
	if grep -q "OC_VersionString = '14\.0\.[0-9]" $STORAGE_ROOT/owncloud/version.php; then
	echo "We are running Nextcloud 14.0.x, upgrading to Nextcloud 15.0.2 first"
	    InstallNextcloud 15.0.2 a54ad9f1b143a5e7c89bf0fae3ba398900266d58
	fi

    fi

    InstallNextcloud $owncloud_ver $owncloud_hash
fi

### Configuring Nextcloud

#touch ./nextcloud_permissions.sh
cat >> ./nextcloud_permissions.sh <<EOF
#!/bin/bash

ocpath='$STORAGE_ROOT/owncloud'
htuser='www-data'
htgroup='www-data'
rootuser='root'

printf "Creating possible missing Directories\n"

mkdir -p \$ocpath/data
mkdir -p \$ocpath/assets
mkdir -p \$ocpath/updater

printf "chmod Files and Directories\n"

find \${ocpath}/ -type f -print0 | xargs -0 chmod 0640
find \${ocpath}/ -type d -print0 | xargs -0 chmod 0770

printf "chown Directories\n"

chown -R \${rootuser}:\${htgroup} \${ocpath}/
chown -R \${htuser}:\${htgroup} \${ocpath}/apps/
chown -R \${htuser}:\${htgroup} \${ocpath}/assets/
chown -R \${htuser}:\${htgroup} \${ocpath}/config/
chown -R \${htuser}:\${htgroup} \${ocpath}/data/
chown -R \${htuser}:\${htgroup} \${ocpath}/themes/
chown -R \${htuser}:\${htgroup} \${ocpath}/updater/

chmod +x \${ocpath}/occ

printf "chmod/chown .htaccess\n"

if [ -f \${ocpath}/.htaccess ]; then
    chmod 0644 \${ocpath}/.htaccess
    chown \${rootuser}:\${htgroup} \${ocpath}/.htaccess
fi

if [ -f \${ocpath}/data/.htaccess ]; then
    chmod 0644 \${ocpath}/data/.htaccess
    chown \${rootuser}:\${htgroup} \${ocpath}/data/.htaccess
fi

EOF

chmod +x ./nextcloud_permissions.sh
./nextcloud_permissions.sh

# Setup Nextcloud if the Nextcloud database does not yet exist. Running setup when
# the database does exist wipes the database and user data.
if [ ! -f $STORAGE_ROOT/owncloud/owncloud.db ]; then
    # Create user data directory
    #mkdir -p $STORAGE_ROOT/owncloud

    # Create an initial configuration file.
    #instanceid=oc$(echo $PRIMARY_HOSTNAME | sha1sum | fold -w 10 | head -n 1)
    instanceid=oc$(echo wolftronics.hu | sha1sum | fold -w 10 | head -n 1)

#    cat << "EOF" > $STORAGE_ROOT/owncloud/config.php
#    cat > $STORAGE_ROOT/owncloud/config.php <<EOF
#<?php
#\$CONFIG = array (
#  'datadirectory' => '$STORAGE_ROOT/owncloud-data/',
#  'instanceid' => '$instanceid',
#  'forcessl' => true, # if unset/false, Nextcloud sends a HSTS=0 header, which conflicts with nginx config
#  'overwritewebroot' => '/cloud',
#  'overwrite.cli.url' => '/cloud',
#  'user_backends' => array(
#    array(
#      'class'=>'OC_User_IMAP',
#      'arguments'=>array('{127.0.0.1:993/imap/ssl/novalidate-cert}')
#    )
#  ),
#  'memcache.local' => '\OC\Memcache\APCu',
#  'mail_smtpmode' => 'sendmail',
#  'mail_smtpsecure' => '',
#  'mail_smtpauthtype' => 'LOGIN',
#  'mail_smtpauth' => false,
#  'mail_smtphost' => '',
#  'mail_smtpport' => '',
#  'mail_smtpname' => '',
#  'mail_smtppassword' => '',
#  'mail_from_address' => 'owncloud',
#);
#?>
#EOF

#ln -sf $STORAGE_ROOT/owncloud/config.php $STORAGE_ROOT/owncloud/config/config.php

echo -n "Creating Nextcloud Setting and Database... "
CFG_CLOUDDBNAME=nextcloud
CFG_CLOUDDBUSER=nextcloud
CFG_CLOUDDBPWD=$(dd if=/dev/urandom bs=1 count=40 2>/dev/null | sha1sum | fold -w 30 | head -n 1)
dbruser=root
CFG_CLOUDADMINUSER=root
CFG_CLOUDADMINPWD=$(dd if=/dev/urandom bs=1 count=40 2>/dev/null | sha1sum | fold -w 30 | head -n 1)

  case $CFG_CLOUDDBTYPE in
    "mysql")
	if [ $CFG_CLOUDDBHOST == "localhost" ]; then
		mysql -u $dbruser -p$CFG_MYSQL_ROOT_PWD -e "CREATE DATABASE $CFG_CLOUDDBNAME" > /dev/null 2>&1
		#mysql -u $dbruser -p$CFG_MYSQL_ROOT_PWD -e "CREATE DATABASE $CFG_CLOUDDBNAME CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci" > /dev/null 2>&1
		mysql -u $dbruser -p$CFG_MYSQL_ROOT_PWD -e "USE $CFG_CLOUDDBNAME" > /dev/null 2>&1
		mysql -u $dbruser -p$CFG_MYSQL_ROOT_PWD -e "GRANT ALL PRIVILEGES ON $CFG_CLOUDDBNAME.* TO '$CFG_CLOUDDBUSER'@'localhost' IDENTIFIED BY '$CFG_CLOUDDBPWD'" > /dev/null 2>&1
		mysql -u $dbruser -p$CFG_MYSQL_ROOT_PWD -e "flush privileges;" > /dev/null 2>&1
	else
		mysql -u $dbruser -p$CFG_MYSQL_ROOT_PWD -h $CFG_CLOUDDBHOST -e "CREATE DATABASE $CFG_CLOUDDBNAME" > /dev/null 2>&1
		mysql -u $dbruser -p$CFG_MYSQL_ROOT_PWD -h $CFG_CLOUDDBHOST -e "USE $CFG_CLOUDDBNAME" > /dev/null 2>&1
		mysql -u $dbruser -p$CFG_MYSQL_ROOT_PWD -h $CFG_CLOUDDBHOST -e "GRANT ALL PRIVILEGES ON $CFG_CLOUDDBNAME.* TO '$CFG_CLOUDDBUSER'@'%' IDENTIFIED BY '$CFG_CLOUDDBPWD'" > /dev/null 2>&1
		mysql -u $dbruser -p$CFG_MYSQL_ROOT_PWD -e "flush privileges;" > /dev/null 2>&1
	fi
	sudo -u www-data php "$STORAGE_ROOT"/owncloud/occ maintenance:install \
	--data-dir "$STORAGE_ROOT/owncloud-data" \
	--database "mysql" \
	--database-name "$CFG_CLOUDDBNAME" \
	--database-user "$CFG_CLOUDDBUSER" \
	--database-pass "$CFG_CLOUDDBPWD" \
	--admin-user "$CFG_CLOUDADMINUSER" \
	--admin-pass "$CFG_CLOUDADMINPWD"

	echo
	echo "Nextcloud version:"
	sudo -u www-data php "$STORAGE_ROOT"/owncloud/occ status
	sleep 3
	echo

	# Enable UTF8mb4 (4-byte support)
	#databases=$(mysql -u root -p"$CFG_MYSQL_ROOT_PWD" -e "SHOW DATABASES;" | tr -d "| " | grep -v Database)
	#for db in $databases; do
	#    if [[ "$db" != "performance_schema" ]] && [[ "$db" != _* ]] && [[ "$db" != "information_schema" ]];
	#    then
	#	echo "Changing to UTF8mb4 on: $db"
	#	mysql -u root -p"$CFG_MYSQL_ROOT_PWD" -e "ALTER DATABASE $db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"
	#    fi
	#done

	# Repair and set Nextcloud config values
	#mysqlcheck -u root -p"$CFG_MYSQL_ROOT_PWD" --auto-repair --optimize --all-databases
	#sudo -u www-data "$STORAGE_ROOT"/owncloud/occ config:system:set mysql.utf8mb4 --type boolean --value="true"
	#sudo -u www-data "$STORAGE_ROOT"/owncloud/occ maintenance:repair
	# Set SMTP mail
	#sudo -u www-data php "$STORAGE_ROOT"/owncloud/occ config:system:set mail_smtpmode --value="smtp"
	# Set logrotate
	sudo -u www-data php "$STORAGE_ROOT"/owncloud/occ config:system:set log_rotate_size --value="10485760"
	# Added trusted domains
	sudo -u www-data php "$STORAGE_ROOT"/owncloud/occ config:system:set trusted_domains 1 --value=192.168.0.105
	sudo -u www-data php "$STORAGE_ROOT"/owncloud/occ config:system:set trusted_domains 2 --value=cloud.wolftronics.hu

#	echo -n "Creating Autoconfig... "
	# Create an auto-configuration file to fill in database settings
	# when the install script is run. Make an administrator account
	# here or else the install can't finish.
#	cat > $STORAGE_ROOT/owncloud/config/autoconfig.php <<EOF
#<?php
#\$AUTOCONFIG = array (
#  # storage/database
#  'directory' => '$STORAGE_ROOT/owncloud',
#  'dbtype' => 'mysql',
#  'dbname' => "$CFG_CLOUDDBNAME",
#  'dbuser' => "$CFG_CLOUDDBUSER",
#  'dbpass' => "$CFG_CLOUDDBPWD",
#  'dbhost' => "$CFG_CLOUDDBHOST",
#  'dbtableprefix' => "",
#  # create an administrator account with a random password so that
#  # the user does not have to enter anything on first load of Nextcloud
#  'adminlogin'    => '$CFG_CLOUDADMINUSER',
#  'adminpass'     => '$CFG_CLOUDADMINPWD',
#  'trusted_domains' =>
#  array (
#    0 => "cloud.wolftronics.hu",
#  ),
#  #'overwrite.cli.url' => "$url1",
#  #'default_language' => "$default_language",
#);
#);
#?>
#EOF

    ;;
    "sqlite")
#	echo -n "Creating Autoconfig..."
	# Create an auto-configuration file to fill in database settings
	# when the install script is run. Make an administrator account
	# here or else the install can't finish.
#	cat > $STORAGE_ROOT/owncloud/config/autoconfig.php <<EOF
#<?php
#\$AUTOCONFIG = array (
#  # storage/database
#  'directory' => '$STORAGE_ROOT/owncloud',
#  'dbtype' => 'sqlite3',
#  # create an administrator account with a random password so that
#  # the user does not have to enter anything on first load of Nextcloud
#  'adminlogin'    => '$CFG_CLOUDADMINUSER',
#  'adminpass'     => '$CFG_CLOUDADMINPWD',
#);
#?>
#EOF
    ;;
  esac

    # Set permissions
    #chown -R www-data.www-data $STORAGE_ROOT/owncloud $STORAGE_ROOT/owncloud

    # Execute Nextcloud's setup step, which creates the Nextcloud sqlite database.
    # It also wipes it if it exists. And it updates config.php with database
    # settings and deletes the autoconfig.php file.
#    (cd $STORAGE_ROOT/owncloud; sudo -u www-data php $STORAGE_ROOT/owncloud/index.php;)
fi

# Update config.php.
# * trusted_domains is reset to localhost by autoconfig starting with ownCloud 8.1.1,
#   so set it here. It also can change if the box's PRIMARY_HOSTNAME changes, so
#   this will make sure it has the right value.
# * Some settings weren't included in previous versions of Mail-in-a-Box.
# * We need to set the timezone to the system timezone to allow fail2ban to ban
#   users within the proper timeframe
# * We need to set the logdateformat to something that will work correctly with fail2ban
# * mail_domain' needs to be set every time we run the setup. Making sure we are setting 
#   the correct domain name if the domain is being change from the previous setup.
# Use PHP to read the settings file, modify it, and write out the new settings array.
#TIMEZONE=$(cat /etc/timezone)
#CONFIG_TEMP=$(/bin/mktemp)
#php << "EOF" > $CONFIG_TEMP && mv $CONFIG_TEMP $STORAGE_ROOT/owncloud/config.php;
#<?php
#include("$STORAGE_ROOT/owncloud/config.php");
#\$CONFIG['trusted_domains'] = array('wolftronics.hu');
#\$CONFIG['memcache.local'] = '\OC\Memcache\APCu';
#\$CONFIG['overwrite.cli.url'] = '/cloud';
#\$CONFIG['mail_from_address'] = 'administrator'; # just the local part, matches our master administrator address
#\$CONFIG['logtimezone'] = '$TIMEZONE';
#\$CONFIG['logdateformat'] = 'Y-m-d H:i:s';
#\$CONFIG['mail_domain'] = '$PRIMARY_HOSTNAME';
#\$CONFIG['mail_domain'] = 'wolftronics.hu';
#echo "<?php\n\\\$CONFIG = ";
#var_export(\$CONFIG);
#echo ";";
#?>
#EOF


#chown www-data.www-data $STORAGE_ROOT/owncloud/config.php

# Enable/disable apps. Note that this must be done after the Nextcloud setup.
# The firstrunwizard gave Josh all sorts of problems, so disabling that.
# user_external is what allows Nextcloud to use IMAP for login. The contacts
# and calendar apps are the extensions we really care about here.
#hide_output sudo -u www-data php $STORAGE_ROOT/owncloud/console.php app:disable firstrunwizard
#hide_output sudo -u www-data php $STORAGE_ROOT/owncloud/console.php app:enable user_external
#hide_output sudo -u www-data php $STORAGE_ROOT/owncloud/console.php app:enable contacts
#hide_output sudo -u www-data php $STORAGE_ROOT/owncloud/console.php app:enable calendar
#hide_output sudo -u www-data php $STORAGE_ROOT/owncloud/console.php app:enable mail
#hide_output sudo -u www-data php $STORAGE_ROOT/owncloud/console.php app:enable notes
#hide_output sudo -u www-data php $STORAGE_ROOT/owncloud/console.php app:enable tasks
#hide_output sudo -u www-data php $STORAGE_ROOT/owncloud/console.php app:enable gallery
#xmlstarlet edit -L -u "/info/dependencies/nextcloud[@min-version='12'] [@max-version='12']/@min-version" -v 10 /	xmlstarlet edit -L -u "/info/dependencies/nextcloud[@min-version='12'] [@max-version='12']/@min-version" -v 10 $STORAGE_ROOT/owncloud/apps/impersonate/appinfo/info.xml
#hide_output sudo -u www-data php $STORAGE_ROOT/owncloud/console.php app:enable impersonate

# When upgrading, run the upgrade script again now that apps are enabled. It seems like
# the first upgrade at the top won't work because apps may be disabled during upgrade?
# Check for success (0=ok, 3=no upgrade needed).
#sudo -u www-data php $STORAGE_ROOT/owncloud/occ upgrade
#if [ \( $? -ne 0 \) -a \( $? -ne 3 \) ]; then exit 1; fi

# Set PHP FPM values to support large file uploads
# (semicolon is the comment character in this file, hashes produce deprecation warnings)
#sed -i "s/upload_max_filesize =.*/upload_max_filesize = 16G/g" /etc/php/7.0/fpm/php.ini
sed -i "s/upload_max_filesize =.*/upload_max_filesize = 1G/g" /etc/php/7.0/fpm/php.ini
#sed -i "s/post_max_size =.*/post_max_size = 16G/g" /etc/php/7.0/fpm/php.ini
sed -i "s/post_max_size =.*/post_max_size = 1G/g" /etc/php/7.0/fpm/php.ini
sed -i "s/output_buffering =.*/output_buffering = 16384/g" /etc/php/7.0/fpm/php.ini
sed -i "s/memory_limit =.*/memory_limit = 512M/g" /etc/php/7.0/fpm/php.ini
sed -i "s/max_execution_time =.*/max_execution_time = 600/g" /etc/php/7.0/fpm/php.ini
sed -i "s/short_open_tag =.*/short_open_tag = On/g" /etc/php/7.0/fpm/php.ini

# Set Nextcloud recommended opcache settings
#if grep -q opcache.enable=0 /etc/php/7.0/mods-available/apcu.ini; then
#    sed -i "s/;opcache.enable=.*/opcache.enable=1/g" /etc/php/7.0/fpm/php.ini
#fi
sed -i "s/;opcache.enable_cli.*/opcache.enable_cli=1/g" /etc/php/7.0/fpm/php.ini
sed -i "s/;opcache.interned_strings_buffer.*/opcache.interned_strings_buffer=8/g" /etc/php/7.0/fpm/php.ini
sed -i "s/max_accelerated_files=.*/max_accelerated_files=10000/g" /etc/php/7.0/fpm/php.ini
sed -i "s/opcache.memory_consumption=.*/opcache.memory_consumption=128/g" /etc/php/7.0/fpm/php.ini
sed -i "s/;opcache.save_comments=.*/opcache.save_comments=1/g" /etc/php/7.0/fpm/php.ini
sed -i "s/opcache.revalidate_freq=.*/opcache.revalidate_freq=1/g" /etc/php/7.0/fpm/php.ini

# Configure the path environment for php-fpm
sed -i '/env\[PATH/s/;//g' /etc/php/7.0/fpm/pool.d/www.conf

# If apc is explicitly disabled we need to enable it
#if grep -q apc.enabled=0 /etc/php/7.1/mods-available/apcu.ini; then
#if grep -q apc.enabled=1 /etc/php/7.0/mods-available/apcu.ini; then
#    echo "apc.enabled=1" >> /etc/php/7.0/mods-available/apcu.ini
#fi

# Set up a cron job for Nextcloud.
cat >> /etc/cron.hourly/wolftronics-owncloud <<EOF
#!/bin/bash
# Wolftronics server setup script
sudo -u www-data php -f $STORAGE_ROOT/owncloud/cron.php
EOF

chmod +x /etc/cron.hourly/wolftronics-owncloud

# There's nothing much of interest that a user could do as an admin for Nextcloud,
# and there's a lot they could mess up, so we don't make any users admins of Nextcloud.
# But if we wanted to, we would do this:
# ```
# for user in $(tools/mail.py user admins); do
#	 sqlite3 $STORAGE_ROOT/owncloud/owncloud.db "INSERT OR IGNORE INTO oc_group_user VALUES ('admin', '$user')"
# done
# ```

#echo -n "Configuring Redis Cache for nextCloud..."
#hide_output add-apt-repository -y ppa:chris-lea/redis-server
#hide_output apt-get update
#apt_get_quiet upgrade
#apt_install redis-server
#restart_service apache2
#restart_service nginx
#phpenmod redis
#hide_output systemctl start redis-server
#hide_output systemctl enable redis-server
#apt_install php-redis

#cat >> $STORAGE_ROOT/owncloud/config/config.php <<EOF
#'memcache.distributed' => '\OC\Memcache\Redis',
#'memcache.local' => '\OC\Memcache\Redis',
#'memcache.locking' => '\OC\Memcache\Redis',
#'redis' => array(
#     'host' => 'localhost',
#     'port' => 6379,
#     ),
#EOF


###################
##  PERMISSIONS  ##
###################
#if [ $perm == "plesk" ]; then
#	chdir="0755"
#	chfile="0644"
#else
#	chdir="0750"
#	chfile="0640"
#fi



#if [ $perm == "plesk" ]; then
#	#chown ${htuser}:psaserv $ncpath
#else
#	rm -f ./nextcloud_permissions.sh
#fi

#################
##  ENDSCREEN  ##
#################

cat >> /root/wtserver_configuration.txt <<EOF
# Nextcloud Store the passwords
URL     		: $url
Nextcloud Admin   	: $CFG_CLOUDADMINUSER
Nextcloud Password   	: $CFG_CLOUDADMINPWD

Database type		: $CFG_CLOUDDBTYPE
Database name		: $CFG_CLOUDDBNAME
Database user		: $CFG_CLOUDDBUSER
Database password	: $CFG_CLOUDDBPWD

EOF

# Enable PHP modules and restart PHP
  restart_service php7.0-fpm

# owncloud gerate ssl
#service nginx stop
#certbot certonly --renew-by-default --standalone --email wolftronics@wolftronics.hu -d cloud.wolftronics.hu
#certbot certonly --agree-tos --non-interactive --renew-by-default --standalone --email wolftronics@wolftronics.hu -d cloud.wolftronics.hu
#certbot certonly \
#  --agree-tos \
#  --non-interactive \
#  --text \
#  --rsa-key-size 4096 \
#  --email admin@example.com \
#  --webroot-path /var/www/html \
#  --domains "example.com, www.example.com"

#ssl cert generated
service nginx stop
#sh /home/administrator/.acme.sh/acme.sh --issue --standalone -d cloud.wolftronics.hu
restart_service nginx

  echo -e "[${green}DONE${NC}]\n"
}
