#!/usr/bin/env bash
#---------------------------------------------------------------------
# install.sh
#
# ISPConfig 3 system installer
#
# Script: install.sh
# Version: 3.0.2
# Author: Matteo Temporini <wolftronics@wolftronics.hu>
# Description: This script will install all the packages needed to install
# ISPConfig 3 on your server.
#
#
#---------------------------------------------------------------------
#set -e 

COLUMNS=$(tput cols)

#---------------------------------------------------------------------
# Bash Colours
#---------------------------------------------------------------------
red='\033[0;31m'
green='\033[0;32m'
BBlack='\033[1;90m'
BWhite='\033[1;97m'
yellow='\e[0;33m'
bold='\e[1m'
underlined='\e[4m'
NC='\033[0m' # No Color

#---------------------------------------------------------------------
#IDENTATION LVLS
#---------------------------------------------------------------------
IDENTATION_LVL_0="${BWhite}===>${NC}"
IDENTATION_LVL_1="   -"
IDENTATION_LVL_2="     *"
IDENTATION_LVL_3="       *"

#Saving current directory
APWD=$(pwd);

#Those lines are for logging porpuses
mkdir -p $APWD/tmp/{logs,sqls,downloads}
echo -e "$IDENTATION_LVL_0 ${BWhite}Create needed directories${NC}"
echo -e "   - ${BWhite}Temp Folder:${NC} ${BBlack}tmp/${NC} [${green}DONE${NC}]"
echo -e "      - ${BWhite}Logs:${NC} ${BBlack}tmp/logs${NC} [${green}DONE${NC}]"
echo -e "      - ${BWhite}SQLs:${NC} ${BBlack}tmp/sqls${NC} [${green}DONE${NC}]"
echo -e "      - ${BWhite}Downloads:${NC} ${BBlack}tmp/downloads${NC} [${green}DONE${NC}]"

echo -n -e "$IDENTATION_LVL_0 ${BWhite}Setup Logging Files${NC} "
exec > >(tee -i tmp/logs/wt_server_setup_programs.log)
exec 2>&1

PROGRAMS_INSTALL_DOWNLOAD=$PWD/tmp/downloads
PROGRAMS_INSTALL_SQLS=$PWD/tmp/sqls
PROGRAMS_INSTALL_LOG_FILES=$PWD/tmp/logs/wt_server_setup_programs.log
touch $PROGRAMS_INSTALL_LOG_FILES

echo -e "[${green}DONE${NC}]"

#---------------------------------------------------------------------
# Global variables
#---------------------------------------------------------------------
echo -n -e "$IDENTATION_LVL_0 ${BWhite}Setup Global Variable${NC} "
CFG_HOSTNAME_FQDN=$(hostname -f)
WT_BACKTITLE="ISPConfig 3 System Installer By Wolftronics"
echo -e "[${green}DONE${NC}]"

echo -e "$IDENTATION_LVL_0 ${BWhite}Checking your system, please wait...${NC} "

#CFG_HOSTNAME_FQDN=$(hostname -f); # hostname -A
IP_ADDRESS=( $(hostname -I) );
RE_IPv4='^((25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$'
IPv4_ADDRESS=( $(for i in "${IP_ADDRESS[@]}"; do [[ "$i" =~ $RE_IPv4 ]] && echo "$i"; done) )
RE_IPv6='^(([0-9a-fA-F]{1,4}:){7}[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,7}:|([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|[0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|:((:[0-9a-fA-F]{1,4}){1,7}|:)|fe80:(:[0-9a-fA-F]{0,4}){0,4}%[0-9a-zA-Z]+|::(ffff(:0{1,4})?:)?((25[0-5]|(2[0-4]|1?[0-9])?[0-9])\.){3}(25[0-5]|(2[0-4]|1?[0-9])?[0-9])|([0-9a-fA-F]{1,4}:){1,4}:((25[0-5]|(2[0-4]|1?[0-9])?[0-9])\.){3}(25[0-5]|(2[0-4]|1?[0-9])?[0-9]))$'
IPv6_ADDRESS=( $(for i in "${IP_ADDRESS[@]}"; do [[ "$i" =~ $RE_IPv6 ]] && echo "$i"; done) )

#---------------------------------------------------------------------
# Load needed functions
#---------------------------------------------------------------------
source $APWD/functions/01_check_linux.sh
source $APWD/functions/02_check_ipv6.sh
source $APWD/functions/03_check_whiptail.sh
source $APWD/functions/04_measure_duration_and_echo.sh
source $APWD/functions/05_apt_get_commands_wrappers.sh
source $APWD/functions/06_normalize_values.sh

#---------------------------------------------------------------------
# Basic Checks
#---------------------------------------------------------------------
CheckLinux
CheckIPV6
CheckWhiptailAndInstallIfNeed

# Check memory
TOTAL_PHYSICAL_MEM=$(awk '/^MemTotal:/ {print $2}' /proc/meminfo)
TOTAL_SWAP=$(awk '/^SwapTotal:/ {print $2}' /proc/meminfo)
if [ "$TOTAL_PHYSICAL_MEM" -lt 524288 ]; then
	echo "This machine has: $(printf "%'d" $((TOTAL_PHYSICAL_MEM / 1024))) MiB ($(printf "%'d" $((((TOTAL_PHYSICAL_MEM * 1024) / 1000) / 1000))) MB) memory (RAM)."
	echo -e "\n${red}Error: ISPConfig needs more memory to function properly. Please run this script on a machine with at least 512 MiB memory, 1 GiB (1024 MiB) recommended.${NC}" >&2
	exit 1
fi

echo -e "$IDENTATION_LVL_0 ${BWhite}System Checking [${NC} ${green}COMPLETED${NC} ${BWhite}]${NC} "

echo -n -e "$IDENTATION_LVL_0 ${BWhite}Please give us 2 seconds before continue ${NC}... "
sleep 2
echo -e "[${NC} ${green}Thanks!${NC} ${BWhite}]"

MeasureTimeDuration $SCRIPT_EXECUTION_START_TIME


# Adapted from: https://github.com/virtualmin/slib/blob/master/slib.sh#L460
RE='^.+\.localdomain$'
RE1='^.{4,253}$'
RE2='^([[:alnum:]][[:alnum:]\-]{0,61}[[:alnum:]]\.)+[a-zA-Z]{2,63}$'
if [[ $CFG_HOSTNAME_FQDN =~ $RE ]]; then
	echo "The hostname is: $CFG_HOSTNAME_FQDN."
	echo -e "${yellow}Warning: Hostname cannot be *.localdomain.${NC}\n"
elif ! [[ $CFG_HOSTNAME_FQDN =~ $RE1 && $CFG_HOSTNAME_FQDN =~ $RE2 ]]; then
	echo "The hostname is: $CFG_HOSTNAME_FQDN."
	echo -e "${yellow}Warning: Hostname is not a valid fully qualified domain name (FQDN).${NC}\n"
fi
if [[ $CFG_HOSTNAME_FQDN =~ $RE ]] || ! [[ $CFG_HOSTNAME_FQDN =~ $RE1 && $CFG_HOSTNAME_FQDN =~ $RE2 ]]; then
	echo "The IP address is: ${IP_ADDRESS[0]}."
	# Source: https://www.faqforge.com/linux/which-ports-are-used-on-a-ispconfig-3-server-and-shall-be-open-in-the-firewall/
	echo -e "${yellow}Warning: If this system is connected to a router and/or behind a NAT, please be sure that the private (internal) IP address is static before continuing.${NC} For routers, static internal IP addresses are usually assigned via DHCP reservation. See your routers user guide for more info… You will also need to forward some ports depending on what software you choose to install:\n\tTCP Ports\n\t\t20\t- FTP\n\t\t21\t- FTP\n\t\t22\t- SSH/SFTP\n\t\t25\t- Mail (SMTP)\n\t\t53\t- DNS\n\t\t80\t- Web (HTTP)\n\t\t110\t- Mail (POP3)\n\t\t143\t- Mail (IMAP)\n\t\t443\t- Web (HTTPS)\n\t\t465\t- Mail (SMTPS)\n\t\t587\t- Mail (SMTP)\n\t\t993\t- Mail (IMAPS)\n\t\t995\t- Mail (POP3S)\n\t\t3306\t- Database\n\t\t5222\t- Chat (XMPP)\n\t\t8080\t- ISPConfig\n\t\t8081\t- ISPConfig\n\t\t10000\t- ISPConfig\n\n\tUDP Ports\n\t\t53\t- DNS\n\t\t3306\t- Database\n" | fold -s -w "$COLUMNS"
	# read -p "Would you like to update the hostname for this system? (recommended) (y/n) " -n 1 -r
	echo -n "Would you like to update the hostname for this system? (recommended) (y/n) "
	read -n 1 -r
	echo -e "\n"   # (optional) move to a new line
	RE='^[Yy]$'
	if [[ $REPLY =~ $RE ]]; then
		while ! [[ $line =~ $RE1 && $line =~ $RE2 ]]; do
			# read -p "Please enter a fully qualified domain name (FQDN) (e.g. ${HOSTNAME%%.*}.example.com): " -r line
			echo -n "Please enter a fully qualified domain name (FQDN) (e.g. ${HOSTNAME%%.*}.example.com): "
			read -r line
		done
		# hostnamectl set-hostname "$line"
		#subdomain=${line%%.*}
		hostnamectl set-hostname "$line"
		if grep -q "^${IP_ADDRESS[0]}" /etc/hosts; then
			sed -i "s/^${IP_ADDRESS[0]}.*/${IP_ADDRESS[0]}\t$line\t$subdomain/" /etc/hosts
		else
			sed -i "s/^127.0.1.1.*/${IP_ADDRESS[0]}\t$line\t$subdomain/" /etc/hosts
		fi
		CFG_HOSTNAME_FQDN=$(hostname -f); # hostname full
	fi
fi

CFG_HOSTNAME_DOMAIN=$(hostname -d); # hostname domain  xxx.hu
CFG_HOSTNAME_DOMAIN_NAME=$(echo "$(hostname -f);" | cut -d"." -f2) # hostname domain xxx
#echo "$(hostname -f);" | cut -d"." -f2

echo
echo -n -e "$IDENTATION_LVL_0 ${BWhite}Load needed Modules ${NC} "
#---------------------------------------------------------------------
# Load needed Modules
#---------------------------------------------------------------------
source $APWD/distros/$DISTRO/preinstallcheck.sh
source $APWD/distros/$DISTRO/askquestions.sh

source $APWD/distros/$DISTRO/install_basics.sh
source $APWD/distros/$DISTRO/install_postfix.sh
source $APWD/distros/$DISTRO/install_mysql.sh
source $APWD/distros/$DISTRO/install_mta.sh
source $APWD/distros/$DISTRO/install_antivirus.sh
source $APWD/distros/$DISTRO/install_webserver.sh
source $APWD/distros/$DISTRO/install_hhvm.sh
source $APWD/distros/$DISTRO/install_ftp.sh
source $APWD/distros/$DISTRO/install_quota.sh
source $APWD/distros/$DISTRO/install_bind.sh
source $APWD/distros/$DISTRO/install_gitlab.sh
source $APWD/distros/$DISTRO/install_webstats.sh
source $APWD/distros/$DISTRO/install_owncloud.sh
source $APWD/distros/$DISTRO/install_jailkit.sh
source $APWD/distros/$DISTRO/install_fail2ban.sh
source $APWD/distros/$DISTRO/install_webmail.sh
source $APWD/distros/$DISTRO/install_metronome.sh
source $APWD/distros/$DISTRO/install_ispconfig.sh
source $APWD/distros/$DISTRO/install_fix.sh
source $APWD/distros/$DISTRO/install_sync.sh
source $APWD/distros/$DISTRO/install_extraphp.sh
source $APWD/distros/$DISTRO/install_joomla.sh
source $APWD/distros/$DISTRO/install_letsencrypt.sh

echo -e " [ ${green}DONE${NC} ] "

if [ -f "/root/wtserver_configuration.txt" ]; then
	source /root/wtserver_configuration.txt
else
	INSTALL_STAGE=0
fi

source $APWD/distros/$DISTRO/install_basephp.sh #to remove in feature release
#---------------------------------------------------------------------
# Main program [ main() ]
#	Run the installer
#---------------------------------------------------------------------
clear

echo "Welcome to ISPConfig Setup Script v.0.0.0.1"
echo "This software is developed by Farkas-Griecs Zoltán"
echo "with the support of the community."
echo "You can visit my website at the followings URLs"
echo "http://www.wolftronics.hu"
echo "and contact me with the following information"
echo "contact email/hangout: wolftronics@wolftronics.hu"
echo ""
echo "========================================="
echo "ISPConfig 3 System Installer"
echo "========================================="
echo -e "\nThis script will do a nearly unattended installation of"
echo "all software needed to run ISPConfig 3."
echo "When this script starts running, it will keep going all the way"
echo -e "So, before you continue, please make sure the following checklist is ok:\n"
echo "- This is a clean standard clean installation for supported systems";
echo -e "- Internet connection is working properly\n\n";
echo -e "The detected Linux Distribution is:\t${PRETTY_NAME:-$ID-$VERSION_ID}"
if [ -n "$ID_LIKE" ]; then
	echo -e "Related Linux Distributions:\t\t$ID_LIKE"
fi
CPU=( $(sed -n 's/^model name[[:space:]]*: *//p' /proc/cpuinfo | uniq) )
if [ -n "$CPU" ]; then
	echo -e "Processor (CPU):\t\t\t${CPU[*]}"
fi
CPU_CORES=$(nproc --all)
echo -e "CPU Cores:\t\t\t\t$CPU_CORES"
ARCHITECTURE=$(getconf LONG_BIT)
echo -e "Architecture:\t\t\t\t$HOSTTYPE ($ARCHITECTURE-bit)"
echo -e "Total memory (RAM):\t\t\t$(printf "%'d" $((TOTAL_PHYSICAL_MEM / 1024))) MiB ($(printf "%'d" $((((TOTAL_PHYSICAL_MEM * 1024) / 1000) / 1000))) MB)"
echo -e "Total swap space:\t\t\t$(printf "%'d" $((TOTAL_SWAP / 1024))) MiB ($(printf "%'d" $((((TOTAL_SWAP * 1024) / 1000) / 1000))) MB)"
if command -v lspci >/dev/null; then
	GPU=( $(lspci 2>/dev/null | grep -i 'vga\|3d\|2d' | sed -n 's/^.*: //p') )
fi
if [ -n "$GPU" ]; then
	echo -e "Graphics Processor (GPU):\t\t${GPU[*]}"
fi
echo -e "Computer name:\t\t\t\t$HOSTNAME"
echo -e "Hostname:\t\t\t\t$CFG_HOSTNAME_FQDN"
echo -e "Domain name:\t\t\t\t$CFG_HOSTNAME_DOMAIN"
echo -e "Domain name:\t\t\t\t$CFG_HOSTNAME_DOMAIN_NAME"
if [ -n "$IPv4_ADDRESS" ]; then
	echo -e "IPv4 address$([[ ${#IPv4_ADDRESS[*]} -gt 1 ]] && echo "es"):\t\t\t\t${IPv4_ADDRESS[*]}"
fi
if [ -n "$IPv6_ADDRESS" ]; then
	echo -e "IPv6 address$([[ ${#IPv6_ADDRESS[*]} -gt 1 ]] && echo "es"):\t\t\t\t${IPv6_ADDRESS[*]}"
fi
TIME_ZONE=$(timedatectl 2>/dev/null | grep -i 'time zone\|timezone' | sed -n 's/^.*: //p')
echo -e "Time zone:\t\t\t\t$TIME_ZONE\n"
if CONTAINER=$(systemd-detect-virt -c); then
	echo -e "Virtualization container:\t\t$CONTAINER\n"
fi
if VM=$(systemd-detect-virt -v); then
	echo -e "Virtual Machine (VM) hypervisor:\t$VM\n"
fi
if uname -r | grep -iq "microsoft"; then
	echo -e "${yellow}Warning: The Windows Subsystem for Linux (WSL) is not yet fully supported by this script.${NC}"
	echo -e "For more information, see this issue: https://github.com/servisys/ispconfig_setup/issues/176\n"
fi
if [ -n "$DISTRO" ]; then
	echo -e "Installing for this Linux Distribution:\t$DISTRO"
	# read -p "Is this correct? (y/n) " -n 1 -r
	echo -n "Is this correct? (y/n) "
	read -n 1 -r
	echo -e "\n"    # (optional) move to a new line
	RE='^[Yy]$'
	if [[ ! $REPLY =~ $RE ]]; then
		exit 1
	fi
else
	echo -e "Sorry but your System is not supported by this script, if you want your system supported " >&2
	echo -e "open an issue on GitHub: https://gitlab.wolftronics.hu/wolftronics-linux/ispconfig_setup" >&2
	if echo "$ID" | grep -iq 'debian\|raspbian\|ubuntu\|centos\|opensuse\|fedora'; then
		echo -e "\nIt is possible that this script will work if you manually set the DISTRO variable to a version of $ID that is supported."
	elif [ -n "$ID_LIKE" ] && echo "$ID_LIKE" | grep -iq 'debian\|raspbian\|ubuntu\|centos\|opensuse\|fedora'; then
		echo -e "\nIt is possible that this script will work if you manually set the DISTRO variable to one of the related Linux distributions that is supported."
	fi
	if echo "$ID" | grep -iq "opensuse"; then
		echo -e "\nYou can use the script here temporary: https://gist.github.com/jniltinho/7734f4879c4469b9a47f3d3eb4ff0bfb"
		echo -e "Adjust it accordingly for your version of $ID and this issue: https://git.ispconfig.org/ispconfig/ispconfig3/issues/5074."
	fi
	exit 1
fi

RE='^.*[^[:space:]]+.*$'
#if [ "$INSTALL_STAGE" == "0" ]; then

if [ "$DISTRO" == "debian8" ] || [ "$DISTRO" == "debian9" ] || [ "$DISTRO" == "debian10" ] || [ "$DISTRO" == "debian11" ] || [ "$DISTRO" == "debian12" ]; then
	while [[ ! "$CFG_ISPCVERSION" =~ $RE ]]
	do
		CFG_ISPCVERSION=$(whiptail --title "ISPConfig Version" --backtitle "$WT_BACKTITLE" --nocancel --radiolist "Select ISPConfig Version you want to install" 10 50 2 "Git stable" "(default)" ON "Develop" "" OFF 3>&1 1>&2 2>&3)
	done
fi

if [ "$DISTRO" == "debian10" ]; then
    source ${APWD}/distros/${DISTRO}/install_mailman.sh
fi

if [ "$DISTRO" == "debian8" ]; then
	while [[ ! "$CFG_MULTISERVER" =~ $RE ]]
	do
		CFG_MULTISERVER=$(whiptail --title "MULTISERVER SETUP" --backtitle "$WT_BACKTITLE" --nocancel --radiolist "Would you like to install ISPConfig in a MultiServer Setup?" 10 50 2 "no" "(default)" ON "yes" "" OFF 3>&1 1>&2 2>&3)
	done
else
	CFG_MULTISERVER=no
fi

if [ -f /etc/debian_version ]; then
	PreInstallCheck
	if [ "$CFG_MULTISERVER" == "no" ]; then
		AskQuestions
	else
		source $APWD/distros/$DISTRO/askquestions_multiserver.sh
		AskQuestionsMultiserver
	fi

	InstallBasics
	if [ "$CFG_LASTKERNEL" == "yes" ]; then
		InstallLastKernel
	fi

	if [ "$CFG_SETUP_DB" == "yes" ] || [ "$CFG_MULTISERVER" == "no" ]; then
		InstallSQLServer
	fi

	if [ "$CFG_SETUP_WEB" == "yes" ] || [ "$CFG_MULTISERVER" == "no" ]; then
		InstallWebServer
		if [ "$CFG_FTP" != "none" ]; then
			InstallFTP
		fi
		if [ "$CFG_QUOTA" == "yes" ]; then
			InstallQuota
		fi
		if [ "$CFG_JKIT" == "yes" ]; then
			InstallJailkit
		fi
		#if [ "$CFG_HHVMINSTALL" == "yes" ]; then
		#	InstallHHVM
		#fi
		if [ "$CFG_METRONOME" == "yes" ]; then
			source $APWD/distros/$DISTRO/install_metronome.sh
			InstallMetronome
		fi
		if [ "$CFG_CLOUDINSTALL" == "yes" ]; then
			InstallCloud
		fi
		if [ "$CFG_WEBMAIL" != "none" ]; then
			InstallWebmail
		fi
	else
		InstallBasePhp    #to remove in feature release
	fi
	if [ "$CFG_SETUP_NS" == "yes" ] || [ "$CFG_MULTISERVER" == "no" ]; then
		InstallBind
	fi

	InstallFail2ban

	if [ "$CFG_SETUP_MAIL" == "yes" ] || [ "$CFG_MULTISERVER" == "no" ]; then
		InstallPostfix
		InstallMTA
		InstallAntiVirus
		if [ "$CFG_MAILMAN" == "yes" ]; then
                	InstallMailman
        	fi
	fi

	InstallWebStats

	if [ "$CFG_SETUP_MASTER" == "yes" ] || [ "$CFG_MULTISERVER" == "no" ]; then
		InstallISPConfig
	fi
	#if [ "$CFG_JOOMLA" == "joomla" ]; then
		#InstallJoomla
	#else
		#InstallDrupal
	#fi
	InstallFix
	if [ "$CFG_GITLABINSTALL" == "yes" ]; then
		InstallGitlabCE
	fi
	

	echo -e "\n${green}Well done! ISPConfig installed and configured correctly :D${NC} 😃"
	echo -e "\nNow you can access to your ISPConfig installation at: ${underlined}https://$CFG_HOSTNAME_FQDN:$CFG_ISPONCFIG_PORT${NC} or ${underlined}https://${IP_ADDRESS[0]}:$CFG_ISPONCFIG_PORT${NC}"
	echo -e "The default ISPConfig Username is: ${bold}admin${NC}\n\t      and the Password is: ${bold}$CFG_ISPONCFIG_ADMIN_PASS${NC}"
	echo -e "${yellow}Warning: This is a security risk. Please change the default password after your first login.${NC}"
  #echo "plese run manual certbot and service nginx restart"
  #echo "sudo su"
  #echo "certbot certonly --agree-tos --non-interactive --renew-by-default --standalone --email wolftronics@wolftronics.hu -d server1.wolftronics.hu"
  #echo "certbot certonly --webroot --webroot-path=/usr/local/ispconfig/interface --email wolftronics@wolftronics.hu -d server1.wolftronics.hu"
  #echo "mkdir /usr/local/ispconfig/interface/ssl/self-signed_originals"
  #echo "mv /usr/local/ispconfig/interface/ssl/isp* /usr/local/ispconfig/interface/ssl/self-signed_originals"
  #echo "ln -s /etc/letsencrypt/live/server1.wolftronics.hu/fullchain.pem /usr/local/ispconfig/interface/ssl/ispserver.crt"
  #echo "ln -s /etc/letsencrypt/live/server1.wolftronics.hu/privkey.pem /usr/local/ispconfig/interface/ssl/ispserver.key"
  #echo "certbot certonly --agree-tos --non-interactive --renew-by-default --standalone --email wolftronics@wolftronics.hu -d gitlab.wolftronics.hu"
  #echo "certbot certonly --webroot --webroot-path=/opt/gitlab/embedded/service/gitlab-rails/public --email wolftronics@wolftronics.hu -d git.wolftronics.hu"
  #echo "certbot certonly --agree-tos --non-interactive --renew-by-default --standalone --email wolftronics@wolftronics.hu -d cloud.wolftronics.hu"
  #echo "certbot certonly --webroot --webroot-path=/var/www/owncloud --email wolftronics@wolftronics.hu -d cloud.wolftronics.hu"
  #echo "certbot certonly --agree-tos --non-interactive --renew-by-default --standalone --email wolftronics@wolftronics.hu -d webmail.wolftronics.hu"
  #echo "certbot certonly --webroot --webroot-path=/var/lib/roundcube --email wolftronics@wolftronics.hu -d webmail.wolftronics.hu"
  #echo "service nginx restart"
  
    if [ "$CFG_WEBMAIL" != "none" ]; then
	    if [ "$CFG_WEBMAIL" == "roundcube" ]; then
            if [ "$DISTRO" != "debian8" ]; then
			    echo -e "\n${red}You will need to edit the username and password in /var/lib/roundcube/plugins/ispconfig3_account/config/config.inc.php of the roundcube user, as the one you set in ISPconfig (under System > remote users)${NC}"
		    fi
			echo  -n -e "Webmail is accessibile at: \n https://$CFG_HOSTNAME_FQDN/webmail \n or \n https://$CFG_IPV4/webmail \n"
		else
				echo  -n -e "Webmail is accessibile at: \n http://$CFG_HOSTNAME_FQDN:8081/webmail \n or \n http://$CFG_IPV4:8081/webmail \n"
		fi	    
    fi
	if [ "$CFG_WEBSERVER" == "nginx" ]; then
		if [ "$CFG_PHPMYADMIN" == "yes" ]; then
			echo "phpMyAdmin is accessible at: http://$CFG_HOSTNAME_FQDN:$CFG_ISPONCFIG_APPS_PORT/phpmyadmin or http://${IP_ADDRESS[0]}:$CFG_ISPONCFIG_APPS_PORT/phpmyadmin";
		fi
		if [ "$DISTRO" == "debian8" ] && [ "$CFG_WEBMAIL" == "roundcube" ]; then
			echo "Webmail is accessible at: https://$CFG_HOSTNAME_FQDN/webmail or https://${IP_ADDRESS[0]}/webmail";
		else
			echo "Webmail is accessible at: http://$CFG_HOSTNAME_FQDN:$CFG_ISPONCFIG_APPS_PORT/webmail or http://${IP_ADDRESS[0]}:$CFG_ISPONCFIG_APPS_PORT/webmail";
		fi
	fi
elif [ -f /etc/redhat-release ]; then # /etc/centos-release
	echo "Attention please, this is the very first version of the script for CentOS $VERSION_ID"
	echo "Please use only for test purpose for now."
	echo -e "${red}Not yet implemented: courier, nginx support${NC}"
	echo -e "${green}Implemented: apache, mysql, bind, postfix, dovecot, roundcube webmail support${NC}"
	echo "Help us to test and implement, press ENTER if you understand what I'm talking about..."
	read DUMMY
	source $APWD/distros/$DISTRO/install_mailman.sh
	PreInstallCheck
	AskQuestions
	InstallBasics
	InstallPostfix
	if [ "$CFG_MAILMAN" == "yes" ]; then
		InstallMailman
	fi
	InstallSQLServer
	InstallMTA
	InstallAntiVirus
	InstallWebServer
	InstallFTP
	if [ "$CFG_QUOTA" == "yes" ]; then
		InstallQuota
	fi
	InstallBind
	InstallWebStats
	if [ "$CFG_JKIT" == "yes" ]; then
		InstallJailkit
	fi
	InstallFail2ban
	if [ "$CFG_METRONOME" == "yes" ]; then
		source $APWD/distros/$DISTRO/install_metronome.sh
		Installmetronome
	fi
	InstallWebmail
	InstallISPConfig
	#InstallFix
	echo -e "\n\n"
	echo -e "\n${green}Well done! ISPConfig installed and configured correctly :D${NC} 😃"
	echo -e "\nNow you can access to your ISPConfig installation at: ${underlined}https://$CFG_HOSTNAME_FQDN:$CFG_ISPONCFIG_PORT${NC} or ${underlined}https://${IP_ADDRESS[0]}:$CFG_ISPONCFIG_PORT${NC}"
	echo -e "The default ISPConfig Username is: ${bold}admin${NC}\n\t      and the Password is: ${bold}$CFG_ISPONCFIG_ADMIN_PASS${NC}"
	echo -e "${yellow}Warning: This is a security risk. Please change the default password after your first login.${NC}"
	echo -e "\n${red}If you setup Roundcube webmail go to: http://$CFG_HOSTNAME_FQDN/roundcubemail/installer and configure db connection${NC}"
	echo -e "${red}After that disable access to installer in /etc/httpd/conf.d/roundcubemail.conf${NC}"
elif [ -f /etc/SuSE-release ]; then
	echo -e "${red}Unsupported Linux distribution.${NC}" >&2
else
	echo -e "${red}Unsupported Linux distribution.${NC}" >&2
fi

#else
#install stage update menu etc..
#  if [ "$INSTALL_STAGE" == "1" ]; then
#	InstallMirrorstage2
#  fi
#  if [ "$INSTALL_STAGE" == "4" ]; then
#     if [ "$CFG_SETUP_MASTER" == "no" ] && [ "$CFG_MIRRORSERVER" == "yes" ]; then
#        InstallUnison
#        sed -i "s/PermitRootLogin yes/PermitRootLogin without-password/g" /etc/ssh/sshd_config
#        restart_service ssh 	
#     fi
#     case $CFG_MTA in
#	"courier")
#	  ;;
	  
#	"dovecot")
#	 InstallMirrorDovecotSync
#	  ;;
#  fi
#fi

exit 0

